import React, { useState, useEffect } from 'react';
import { ContractList } from './ContractList';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { RefreshCw, AlertCircle, FileText, Plus } from 'lucide-react';

interface Contract {
  id: string;
  title: string;
  type: string;
  status: 'draft' | 'review' | 'approval' | 'active' | 'completed';
  property?: string;
  property_location?: string;
  property_type?: string;
  bedrooms?: number;
  bathrooms?: number;
  built_up_area?: number;
  sale_price?: number;
  monthly_rent?: number;
  property_email?: string;
  property_owner?: string;
  parties: string[];
  party_emails?: string[];
  amount?: number;
  start_date?: string;
  end_date?: string;
  created_at: string;
  next_milestone?: string;
  progress: number;
}

export const ContractListWrapper: React.FC = () => {
  const [contracts, setContracts] = useState<Contract[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [refreshing, setRefreshing] = useState(false);

  const fetchContracts = async (isRefresh = false) => {
    try {
      if (isRefresh) {
        setRefreshing(true);
      } else {
        setLoading(true);
      }
      setError(null);
      
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      if (!token) {
        setContracts([]);
        setLoading(false);
        setRefreshing(false);
        return;
      }

      // First check if backend is available
      try {
        const healthResponse = await fetch('http://localhost:8090/api/health');
        if (!healthResponse.ok) {
          throw new Error('Backend not available');
        }
      } catch (healthError) {
        console.warn('Backend health check failed:', healthError);
        setError('Backend service is not available. Please ensure the backend server is running.');
        setContracts([]);
        setLoading(false);
        setRefreshing(false);
        return;
      }

      const response = await fetch('http://localhost:8090/api/contracts/', {
        headers: {
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        }
      });
      
      if (response.ok) {
        const contractsData = await response.json();
        console.log('Contracts fetched successfully:', contractsData);
        setContracts(Array.isArray(contractsData) ? contractsData : []);
      } else if (response.status === 404) {
        // Endpoint not implemented yet, show empty state
        console.log('Contracts endpoint not yet implemented');
        setContracts([]);
      } else if (response.status === 401) {
        setError('Session expired. Please login again.');
        localStorage.removeItem('access_token');
        localStorage.removeItem('sellerToken');
      } else {
        console.error('Error fetching contracts:', response.status);
        setContracts([]);
      }
    } catch (error) {
      console.error('Error fetching contracts:', error);
      setContracts([]);
      if (error instanceof TypeError && error.message.includes('fetch')) {
        setError('Cannot connect to backend server. Please check if the backend is running on http://localhost:8090');
      } else {
        setError('Failed to load contracts. Please check your connection.');
      }
    } finally {
      setLoading(false);
      setRefreshing(false);
    }
  };

  useEffect(() => {
    fetchContracts();
  }, []);

  if (loading) {
    return (
      <Card className="bg-white shadow-sm">
        <CardContent className="p-6">
          <div className="text-center py-8">
            <RefreshCw className="w-8 h-8 animate-spin mx-auto mb-4 text-gray-400" />
            <p className="text-gray-500">Loading contracts...</p>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (error) {
    return (
      <Card className="bg-white shadow-sm">
        <CardContent className="p-6">
          <div className="text-center py-8">
            <AlertCircle className="w-12 h-12 text-red-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">Error Loading Contracts</h3>
            <p className="text-red-600 mb-4">{error}</p>
            <Button 
              onClick={() => fetchContracts(true)}
              variant="outline"
              className="text-red-600 border-red-300 hover:bg-red-50"
            >
              <RefreshCw className="w-4 h-4 mr-2" />
              Try Again
            </Button>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (contracts.length === 0) {
    return (
      <Card className="bg-white shadow-sm">
        <CardHeader className="border-b border-gray-200">
          <div className="flex justify-between items-center">
            <CardTitle className="text-xl font-semibold text-gray-900">
              My Contracts
            </CardTitle>
            <Button 
              variant="outline" 
              onClick={() => fetchContracts(true)}
              disabled={refreshing}
              className="flex items-center gap-2"
            >
              <RefreshCw className={`w-4 h-4 ${refreshing ? 'animate-spin' : ''}`} />
              {refreshing ? 'Refreshing...' : 'Refresh'}
            </Button>
          </div>
        </CardHeader>
        <CardContent className="p-6">
          <div className="text-center py-12">
            <div className="mx-auto w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mb-4">
              <FileText className="w-12 h-12 text-gray-400" />
            </div>
            <h3 className="text-lg font-medium text-gray-900 mb-2">No contracts found</h3>
            <p className="text-gray-500 mb-6">
              You don't have any contracts yet. Create your first contract to get started.
            </p>
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card className="bg-white shadow-sm">
      <CardHeader className="border-b border-gray-200">
        <div className="flex justify-between items-center">
          <CardTitle className="text-xl font-semibold text-gray-900">
            My Contracts ({contracts.length})
          </CardTitle>
          <Button 
            variant="outline" 
            onClick={() => fetchContracts(true)}
            disabled={refreshing}
            className="flex items-center gap-2"
          >
            <RefreshCw className={`w-4 h-4 ${refreshing ? 'animate-spin' : ''}`} />
            {refreshing ? 'Refreshing...' : 'Refresh'}
          </Button>
        </div>
      </CardHeader>
      <CardContent className="p-6">
        <ContractList 
          contracts={contracts} 
          onRefresh={() => fetchContracts(true)}
        />
      </CardContent>
    </Card>
  );
};