import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { ESignPad } from './ESignPad';
import { CheckCircle, Clock, FileText, Download, AlertCircle, User, Building } from 'lucide-react';

interface ContractSigningModalProps {
  isOpen: boolean;
  onClose: () => void;
  contractId: string;
  onSigningComplete: () => void;
}

interface SigningStatus {
  contract_id: string;
  user_role: 'party_1' | 'party_2';
  party_1_signed: boolean;
  party_2_signed: boolean;
  ready_for_signing: boolean;
  contract_details: {
    contract_number: string;
    contract_amount: number;
    contract_terms: string;
    property_id: number;
  };
}

interface SigningResult {
  signature_id: string;
  contract_fully_signed: boolean;
  stamp_paper_generated?: boolean;
  stamp_paper?: {
    certificate_number: string;
    stamp_duty_amount: number;
    pdf_data: string;
    verification_url: string;
  };
  waiting_for?: string;
  message: string;
}

export const ContractSigningModal: React.FC<ContractSigningModalProps> = ({
  isOpen,
  onClose,
  contractId,
  onSigningComplete
}) => {
  const [step, setStep] = useState<'loading' | 'review' | 'signing' | 'completed' | 'error'>('loading');
  const [signingStatus, setSigningStatus] = useState<SigningStatus | null>(null);
  const [signingResult, setSigningResult] = useState<SigningResult | null>(null);
  const [error, setError] = useState<string | null>(null);
  const [isSubmitting, setIsSubmitting] = useState(false);

  useEffect(() => {
    if (isOpen && contractId) {
      initiateSigning();
    }
  }, [isOpen, contractId]);

  const initiateSigning = async () => {
    try {
      setStep('loading');
      setError(null);

      // Get user ID from localStorage
      let userId = null;
      try {
        // Try seller-specific storage first
        userId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
        
        // Try to get from sellerToken
        if (!userId) {
          const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
          if (sellerToken && sellerToken.includes('.')) {
            try {
              const payload = JSON.parse(atob(sellerToken.split('.')[1]));
              userId = payload.user_id || payload.sub || payload.id;
            } catch (e) {
              console.warn('Failed to decode token');
            }
          }
        }
        
        // Try user object
        if (!userId) {
          const user = JSON.parse(localStorage.getItem('user') || '{}');
          userId = user.user_id || user.id;
        }
        
        // Try sellerData object
        if (!userId) {
          const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
          userId = sellerData.user_id || sellerData.id;
        }
        
        if (!userId) {
          throw new Error('Please log in to sign contracts');
        }
        
        console.log('Initiating signing with user ID:', userId);
      } catch (e) {
        throw new Error('Failed to get user information. Please log in again.');
      }

      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/initiate-signing?user_id=${userId}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.detail || 'Failed to initiate signing');
      }

      const data = await response.json();
      setSigningStatus(data);
      setStep('review');
    } catch (err: any) {
      setError(err.message);
      setStep('error');
    }
  };

  const handleSign = async (signatureData: string) => {
    if (!signingStatus) return;

    try {
      setIsSubmitting(true);
      setError(null);

      // Get user ID from localStorage
      let userId = null;
      try {
        // Try seller-specific storage first
        userId = localStorage.getItem('sellerId') || localStorage.getItem('user_id');
        
        // Try to get from sellerToken
        if (!userId) {
          const sellerToken = localStorage.getItem('sellerToken') || localStorage.getItem('access_token');
          if (sellerToken && sellerToken.includes('.')) {
            try {
              const payload = JSON.parse(atob(sellerToken.split('.')[1]));
              userId = payload.user_id || payload.sub || payload.id;
            } catch (e) {
              console.warn('Failed to decode token');
            }
          }
        }
        
        // Try user object
        if (!userId) {
          const user = JSON.parse(localStorage.getItem('user') || '{}');
          userId = user.user_id || user.id;
        }
        
        // Try sellerData object
        if (!userId) {
          const sellerData = JSON.parse(localStorage.getItem('sellerData') || '{}');
          userId = sellerData.user_id || sellerData.id;
        }
        
        if (!userId) {
          throw new Error('Please log in to sign contracts');
        }
        
        console.log('Signing with user ID:', userId);
      } catch (e) {
        throw new Error('Failed to get user information. Please log in again.');
      }

      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/sign-digital?user_id=${userId}`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          signature_data: signatureData,
          device_info: {
            userAgent: navigator.userAgent,
            timestamp: new Date().toISOString()
          }
        })
      });

      if (!response.ok) {
        const errorData = await response.json();
        
        // Handle "already signed" case as success, not error
        if (errorData.detail && errorData.detail.toLowerCase().includes('already signed')) {
          setSigningResult({
            signature_id: 'existing',
            contract_fully_signed: false,
            message: '✅ You have already signed this contract. Waiting for the other party to complete their signature.',
            waiting_for: 'other party'
          });
          setStep('completed');
          onSigningComplete();
          return;
        }
        
        throw new Error(errorData.detail || 'Failed to sign contract');
      }

      const result = await response.json();
      setSigningResult(result);
      setStep('completed');
      onSigningComplete();
    } catch (err: any) {
      setError(err.message);
      setStep('error');
    } finally {
      setIsSubmitting(false);
    }
  };

  const downloadStampPaper = async () => {
    if (!signingResult?.stamp_paper) return;

    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `stamp_paper_${signingResult.stamp_paper.certificate_number}.pdf`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
      }
    } catch (err) {
      console.error('Failed to download stamp paper:', err);
    }
  };

  const renderLoadingStep = () => (
    <div className="text-center py-8">
      <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
      <p className="text-gray-600">Preparing contract for signing...</p>
    </div>
  );

  const renderReviewStep = () => {
    if (!signingStatus) return null;

    return (
      <div className="space-y-6">
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-center gap-2 mb-2">
            <FileText className="w-5 h-5 text-blue-600" />
            <h3 className="font-semibold text-blue-900">Contract Details</h3>
          </div>
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div>
              <span className="font-medium">Contract Number:</span>
              <p className="text-gray-700">{signingStatus.contract_details.contract_number}</p>
            </div>
            <div>
              <span className="font-medium">Amount:</span>
              <p className="text-gray-700">₹{signingStatus.contract_details.contract_amount.toLocaleString()}</p>
            </div>
          </div>
        </div>

        <div className="bg-gray-50 border border-gray-200 rounded-lg p-4">
          <h3 className="font-semibold mb-3">Signing Status</h3>
          <div className="space-y-2">
            <div className="flex items-center gap-3">
              <div className="flex items-center gap-2">
                <User className="w-4 h-4" />
                <span className="text-sm">Party 1</span>
              </div>
              <Badge variant={signingStatus.party_1_signed ? "default" : "secondary"}>
                {signingStatus.party_1_signed ? (
                  <>
                    <CheckCircle className="w-3 h-3 mr-1" />
                    Signed
                  </>
                ) : (
                  <>
                    <Clock className="w-3 h-3 mr-1" />
                    Pending
                  </>
                )}
              </Badge>
            </div>
            <div className="flex items-center gap-3">
              <div className="flex items-center gap-2">
                <Building className="w-4 h-4" />
                <span className="text-sm">Party 2</span>
              </div>
              <Badge variant={signingStatus.party_2_signed ? "default" : "secondary"}>
                {signingStatus.party_2_signed ? (
                  <>
                    <CheckCircle className="w-3 h-3 mr-1" />
                    Signed
                  </>
                ) : (
                  <>
                    <Clock className="w-3 h-3 mr-1" />
                    Pending
                  </>
                )}
              </Badge>
            </div>
          </div>
        </div>

        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex items-start gap-2">
            <AlertCircle className="w-5 h-5 text-yellow-600 mt-0.5" />
            <div>
              <h4 className="font-medium text-yellow-900">Your Role: {signingStatus.user_role.toUpperCase().replace('_', ' ')}</h4>
              <p className="text-sm text-yellow-800 mt-1">
                By signing this contract, you agree to all terms and conditions. 
                Once both parties sign, a digital stamp paper will be automatically generated.
              </p>
            </div>
          </div>
        </div>

        <div className="flex gap-3 justify-end">
          <Button variant="outline" onClick={onClose}>
            Cancel
          </Button>
          <Button 
            onClick={() => setStep('signing')} 
            style={{ backgroundColor: '#2563eb', color: 'white', border: 'none' }}
          >
            Proceed to Sign
          </Button>
        </div>
      </div>
    );
  };

  const renderSigningStep = () => (
    <div className="space-y-4">
      <div className="text-center mb-6">
        <h3 className="text-lg font-semibold mb-2">Digital Signature Required</h3>
        <p className="text-gray-600">Please sign in the pad below to complete the contract</p>
      </div>
      
      <ESignPad
        onSign={handleSign}
        onCancel={() => setStep('review')}
      />
      
      {isSubmitting && (
        <div className="text-center py-4">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto mb-2"></div>
          <p className="text-gray-600">Processing signature...</p>
        </div>
      )}
    </div>
  );

  const renderCompletedStep = () => {
    if (!signingResult) return null;

    return (
      <div className="space-y-6">
        <div className="text-center">
          <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
            <CheckCircle className="w-8 h-8 text-green-600" />
          </div>
          <h3 className="text-xl font-semibold text-green-900 mb-2">
            {signingResult.contract_fully_signed ? 'Contract Fully Executed!' : 'Signature Recorded'}
          </h3>
          <p className="text-gray-600">{signingResult.message}</p>
        </div>

        {signingResult.contract_fully_signed && signingResult.stamp_paper && (
          <Card className="bg-green-50 border-green-200">
            <CardHeader>
              <CardTitle className="text-green-900 flex items-center gap-2">
                <FileText className="w-5 h-5" />
                Digital Stamp Paper Generated
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="grid grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="font-medium">Certificate Number:</span>
                  <p className="text-gray-700">{signingResult.stamp_paper.certificate_number}</p>
                </div>
                <div>
                  <span className="font-medium">Stamp Duty:</span>
                  <p className="text-gray-700">₹{signingResult.stamp_paper.stamp_duty_amount.toFixed(2)}</p>
                </div>
              </div>
              
              <div className="flex gap-3">
                <Button onClick={downloadStampPaper} className="flex items-center gap-2">
                  <Download className="w-4 h-4" />
                  Download Stamp Paper
                </Button>
                <Button 
                  variant="outline" 
                  onClick={() => window.open(signingResult.stamp_paper!.verification_url, '_blank')}
                >
                  Verify Online
                </Button>
              </div>
            </CardContent>
          </Card>
        )}

        {!signingResult.contract_fully_signed && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-center gap-2">
              <Clock className="w-5 h-5 text-yellow-600" />
              <span className="font-medium text-yellow-900">
                Waiting for {signingResult.waiting_for} signature
              </span>
            </div>
            <p className="text-sm text-yellow-800 mt-1">
              The contract will be fully executed once the {signingResult.waiting_for} signs.
            </p>
          </div>
        )}

        <div className="flex justify-end">
          <Button onClick={onClose}>
            Close
          </Button>
        </div>
      </div>
    );
  };

  const renderErrorStep = () => (
    <div className="text-center py-8">
      <div className="w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4">
        <AlertCircle className="w-8 h-8 text-red-600" />
      </div>
      <h3 className="text-lg font-semibold text-red-900 mb-2">Error</h3>
      <p className="text-red-700 mb-4">{error}</p>
      <div className="flex gap-3 justify-center">
        <Button variant="outline" onClick={onClose}>
          Close
        </Button>
        <Button onClick={initiateSigning}>
          Try Again
        </Button>
      </div>
    </div>
  );

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-2xl max-h-[90vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle>Contract Digital Signing</DialogTitle>
          <DialogDescription>
            Complete the digital signing process for your contract
          </DialogDescription>
        </DialogHeader>
        
        <div className="py-4">
          {step === 'loading' && renderLoadingStep()}
          {step === 'review' && renderReviewStep()}
          {step === 'signing' && renderSigningStep()}
          {step === 'completed' && renderCompletedStep()}
          {step === 'error' && renderErrorStep()}
        </div>
      </DialogContent>
    </Dialog>
  );
};