import React, { useState, useEffect } from 'react';
import { contractsApi } from '@/api/contracts';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { ESignPad } from './ESignPad';
import { toast } from '@/utils/toast';
import jsPDF from 'jspdf';

interface ContractViewerProps {
  contractId: string;
}

export const ContractViewer: React.FC<ContractViewerProps> = ({ contractId }) => {
  const [contract, setContract] = useState<any>(null);
  const [signatures, setSignatures] = useState<any[]>([]);
  const [showSignPad, setShowSignPad] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    loadContract();
    loadSignatures();
  }, [contractId]);

  const loadContract = async () => {
    try {
      const data = await contractsApi.getById(contractId);
      setContract(data);
    } catch (error) {
      toast.error('Failed to load contract');
    } finally {
      setLoading(false);
    }
  };

  const loadSignatures = async () => {
    try {
      const data = await contractsApi.getSignatures(contractId);
      setSignatures(data);
    } catch (error) {
      console.error('Failed to load signatures');
    }
  };

  const handleSign = async (signatureData: string) => {
    try {
      const userStr = localStorage.getItem('user');
      if (!userStr) return;

      const user = JSON.parse(userStr);
      const role = user.user_id === contract.buyer_id ? 'buyer' : 'seller';

      await contractsApi.sign(contractId, {
        signer_role: role,
        signature_data: signatureData,
        ip_address: await fetch('https://api.ipify.org?format=json').then(r => r.json()).then(d => d.ip).catch(() => null)
      });

      toast.success('Contract signed successfully');
      setShowSignPad(false);
      await loadSignatures();
      await loadContract();
      
      // Check if all parties signed and create escrow
      const updatedSigs = await contractsApi.getSignatures(contractId);
      const updatedContract = await contractsApi.getById(contractId);
      
      if (updatedContract.status === 'signed') {
        try {
          const { escrowApi } = await import('@/api/escrow');
          await escrowApi.create({
            contract_id: contractId,
            property_id: contract.property_id,
            buyer_id: contract.buyer_id,
            seller_id: contract.seller_id,
            amount: contract.contract_amount,
            description: `Escrow for contract ${contract.contract_number}`
          });
          toast.success('Contract fully signed. Escrow created automatically.');
          await loadContract();
          await loadSignatures();
        } catch (escrowError) {
          console.error('Escrow creation failed:', escrowError);
          toast.info('Contract signed. Please create escrow manually.');
        }
      }
    } catch (error: any) {
      toast.error(error.message || 'Failed to sign contract');
    }
  };

  const getStatusColor = (status: string) => {
    const colors: any = {
      draft: 'bg-gray-100 text-gray-800',
      pending_signatures: 'bg-yellow-100 text-yellow-800',
      signed: 'bg-green-100 text-green-800',
      active: 'bg-blue-100 text-blue-800',
      completed: 'bg-purple-100 text-purple-800',
      cancelled: 'bg-red-100 text-red-800'
    };
    return colors[status] || 'bg-gray-100 text-gray-800';
  };

  const userHasSigned = () => {
    const userStr = localStorage.getItem('user');
    if (!userStr) return false;
    const user = JSON.parse(userStr);
    return signatures.some(s => s.signer_id === user.user_id);
  };

  const downloadContractPDF = () => {
    const pdf = new jsPDF();
    const pageWidth = pdf.internal.pageSize.getWidth();
    const pageHeight = pdf.internal.pageSize.getHeight();
    const margin = 15;
    let yPos = margin + 5;

    const buyerSig = signatures.find(s => s.signer_role === 'buyer');
    const sellerSig = signatures.find(s => s.signer_role === 'seller');

    // Border
    pdf.setDrawColor(0, 86, 210);
    pdf.setLineWidth(0.5);
    pdf.rect(margin, margin, pageWidth - 2 * margin, pageHeight - 2 * margin);

    // Header
    pdf.setFillColor(0, 86, 210);
    pdf.rect(margin, margin, pageWidth - 2 * margin, 25, 'F');
    pdf.setTextColor(255, 255, 255);
    pdf.setFontSize(18);
    pdf.setFont('helvetica', 'bold');
    pdf.text('PROPERTY PURCHASE CONTRACT', pageWidth / 2, yPos + 15, { align: 'center' });
    
    yPos += 30;
    pdf.setTextColor(0, 0, 0);

    // Contract Info
    pdf.setFontSize(9);
    pdf.setFont('helvetica', 'normal');
    pdf.text(`Contract No: ${contract.contract_number}`, margin + 5, yPos);
    pdf.text(`Date: ${new Date(contract.created_at).toLocaleDateString('en-IN')}`, pageWidth - margin - 40, yPos);
    yPos += 10;

    // Parties Section
    pdf.setFontSize(10);
    pdf.setFont('helvetica', 'bold');
    pdf.text('PARTIES TO THE CONTRACT', margin + 5, yPos);
    yPos += 7;

    pdf.setFontSize(8);
    pdf.setFont('helvetica', 'normal');
    pdf.text(`Buyer: ${buyerSig?.signer_name || 'Pending'}`, margin + 5, yPos);
    yPos += 4;
    pdf.text(`Email: ${buyerSig?.signer_email || 'Pending'}`, margin + 5, yPos);
    yPos += 8;
    pdf.text(`Seller: ${sellerSig?.signer_name || 'Pending'}`, margin + 5, yPos);
    yPos += 4;
    pdf.text(`Email: ${sellerSig?.signer_email || 'Pending'}`, margin + 5, yPos);
    yPos += 10;

    // Property Details
    pdf.setFontSize(10);
    pdf.setFont('helvetica', 'bold');
    pdf.text('PROPERTY DETAILS', margin + 5, yPos);
    yPos += 7;

    pdf.setFontSize(8);
    pdf.setFont('helvetica', 'normal');
    pdf.text(`Property ID: #${contract.property_id}`, margin + 5, yPos);
    yPos += 4;
    pdf.text(`Contract Amount: ₹${contract.contract_amount?.toLocaleString('en-IN')}`, margin + 5, yPos);
    yPos += 12;

    // Terms and Conditions
    pdf.setFontSize(10);
    pdf.setFont('helvetica', 'bold');
    pdf.text('TERMS AND CONDITIONS', margin + 5, yPos);
    yPos += 7;

    pdf.setFontSize(7);
    pdf.setFont('helvetica', 'normal');
    const termsLines = pdf.splitTextToSize(contract.contract_terms, pageWidth - 2 * margin - 10);
    termsLines.slice(0, 15).forEach((line: string) => {
      pdf.text(line, margin + 5, yPos);
      yPos += 3.5;
    });

    yPos += 8;

    // Signatures Section
    pdf.setFontSize(10);
    pdf.setFont('helvetica', 'bold');
    pdf.text('SIGNATURES', margin + 5, yPos);
    yPos += 10;

    // Buyer Signature (Left)
    pdf.setFontSize(8);
    pdf.setFont('helvetica', 'bold');
    pdf.text('BUYER:', margin + 5, yPos);
    yPos += 5;
    
    if (buyerSig?.signature_data) {
      try {
        pdf.addImage(buyerSig.signature_data, 'PNG', margin + 5, yPos, 40, 15);
      } catch (e) {
        pdf.setFont('helvetica', 'italic');
        pdf.text('[Signature Error]', margin + 5, yPos + 7);
      }
    } else {
      pdf.setFont('helvetica', 'italic');
      pdf.text('[Pending Signature]', margin + 5, yPos + 7);
    }
    
    pdf.setFont('helvetica', 'normal');
    pdf.setFontSize(7);
    pdf.text(`Name: ${buyerSig?.signer_name || 'Pending'}`, margin + 5, yPos + 18);
    pdf.text(`Date: ${buyerSig ? new Date(buyerSig.signed_at).toLocaleDateString('en-IN') : 'Pending'}`, margin + 5, yPos + 22);

    // Seller Signature (Right)
    const rightX = pageWidth / 2 + 5;
    pdf.setFontSize(8);
    pdf.setFont('helvetica', 'bold');
    pdf.text('SELLER:', rightX, yPos - 5);
    
    if (sellerSig?.signature_data) {
      try {
        pdf.addImage(sellerSig.signature_data, 'PNG', rightX, yPos, 40, 15);
      } catch (e) {
        pdf.setFont('helvetica', 'italic');
        pdf.text('[Signature Error]', rightX, yPos + 7);
      }
    } else {
      pdf.setFont('helvetica', 'italic');
      pdf.text('[Pending Signature]', rightX, yPos + 7);
    }
    
    pdf.setFont('helvetica', 'normal');
    pdf.setFontSize(7);
    pdf.text(`Name: ${sellerSig?.signer_name || 'Pending'}`, rightX, yPos + 18);
    pdf.text(`Date: ${sellerSig ? new Date(sellerSig.signed_at).toLocaleDateString('en-IN') : 'Pending'}`, rightX, yPos + 22);

    // Footer
    pdf.setFillColor(240, 240, 240);
    pdf.rect(margin, pageHeight - margin - 10, pageWidth - 2 * margin, 10, 'F');
    pdf.setTextColor(60, 60, 60);
    pdf.setFontSize(7);
    pdf.setFont('helvetica', 'normal');
    pdf.text('© 2025 Alstonair Technologies Pvt. Ltd. | NAL India – Confidential Document', pageWidth / 2, pageHeight - margin - 4, { align: 'center' });

    pdf.save(`Contract_${contract.contract_number}.pdf`);
    toast.success('Contract PDF downloaded successfully!');
  };

  if (loading) return <div>Loading...</div>;
  if (!contract) return <div>Contract not found</div>;

  console.log('Contract status:', contract.status);
  console.log('Signatures:', signatures);

  return (
    <div className="space-y-6">
      <Card className="p-6">
        <div className="flex justify-between items-start mb-4">
          <div>
            <h2 className="text-2xl font-bold">{contract.contract_number}</h2>
            <p className="text-gray-600">{contract.contract_type}</p>
          </div>
          <Badge className={getStatusColor(contract.status)}>
            {contract.status.replace('_', ' ').toUpperCase()}
          </Badge>
        </div>

        <div className="grid grid-cols-2 gap-4 mb-6">
          <div>
            <label className="text-sm text-gray-600">Contract Amount</label>
            <p className="text-lg font-semibold">₹{contract.contract_amount?.toLocaleString()}</p>
          </div>
          <div>
            <label className="text-sm text-gray-600">Property ID</label>
            <p className="text-lg font-semibold">#{contract.property_id}</p>
          </div>
        </div>

        <div className="mb-6">
          <h3 className="font-semibold mb-2">Contract Terms</h3>
          <div className="bg-gray-50 p-4 rounded whitespace-pre-wrap">
            {contract.contract_terms}
          </div>
        </div>

        <div className="mb-6">
          <h3 className="font-semibold mb-2">Signatures ({signatures.length})</h3>
          <div className="space-y-2">
            {signatures.map((sig) => (
              <div key={sig.id} className="flex items-center gap-2 p-2 bg-green-50 rounded">
                <span className="text-green-600">✓</span>
                <span>{sig.signer_name || 'User'}</span>
                <span className="text-sm text-gray-600">({sig.signer_role})</span>
                <span className="text-xs text-gray-500 ml-auto">
                  {new Date(sig.signed_at).toLocaleString()}
                </span>
              </div>
            ))}
          </div>
        </div>

        <div className="space-y-2">
          {!userHasSigned() && contract.status !== 'signed' && (
            <Button onClick={() => setShowSignPad(true)} className="w-full">
              Sign Contract
            </Button>
          )}
          
          {signatures.length >= 2 && (
            <>
              <Button 
                onClick={downloadContractPDF}
                className="w-full mb-2"
                style={{ backgroundColor: '#2563eb', color: 'white' }}
              >
                📎 Download Contract PDF
              </Button>
              <Button 
                onClick={async () => {
                  try {
                    const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
                    const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper`, {
                      headers: {
                        'Authorization': `Bearer ${token}`
                      }
                    });

                    if (response.ok) {
                      const blob = await response.blob();
                      const url = window.URL.createObjectURL(blob);
                      const a = document.createElement('a');
                      a.href = url;
                      a.download = `official_stamp_paper_${contract.contract_number}.pdf`;
                      document.body.appendChild(a);
                      a.click();
                      window.URL.revokeObjectURL(url);
                      document.body.removeChild(a);
                      toast.success('Official stamp paper downloaded successfully!');
                    } else {
                      toast.error('Stamp paper not available yet');
                    }
                  } catch (error) {
                    toast.error('Failed to download stamp paper');
                  }
                }}
                className="w-full mb-2"
                style={{ backgroundColor: '#059669', color: 'white' }}
              >
                🏛️ Download Official Stamp Paper
              </Button>
            </>
          )}

          {contract.status === 'signed' && contract.buyer_id === JSON.parse(localStorage.getItem('user') || '{}').user_id && (
          <Button 
            onClick={async () => {
              try {
                const { escrowApi } = await import('@/api/escrow');
                const escrows = await escrowApi.list();
                const contractEscrow = escrows.find((e: any) => e.contract_id === contractId);
                
                if (contractEscrow) {
                  window.location.href = `/escrow/${contractEscrow.id}`;
                } else {
                  await escrowApi.create({
                    contract_id: contractId,
                    property_id: contract.property_id,
                    buyer_id: contract.buyer_id,
                    seller_id: contract.seller_id,
                    escrow_amount: contract.contract_amount,
                    description: `Escrow for contract ${contract.contract_number}`
                  });
                  toast.success('Escrow created! Reloading...');
                  setTimeout(() => window.location.reload(), 1000);
                }
              } catch (error) {
                toast.error('Failed to create escrow');
              }
            }}
            className="w-full mt-2"
            style={{ backgroundColor: '#16a34a' }}
          >
            Deposit Funds to Escrow →
          </Button>
          )}
        </div>
      </Card>

      {showSignPad && (
        <ESignPad
          onSign={handleSign}
          onCancel={() => setShowSignPad(false)}
        />
      )}
    </div>
  );
};
