import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { CheckCircle, Clock, FileText, Users, Shield, PenTool } from 'lucide-react';

interface WorkflowStep {
  stage: string;
  status: 'pending' | 'completed' | 'current';
  completedAt?: string;
  assignedTo?: string;
  comments?: string;
}

interface ContractWorkflowProps {
  workflow: WorkflowStep[];
  currentStage: string;
}

export const ContractWorkflow: React.FC<ContractWorkflowProps> = ({
  workflow,
  currentStage
}) => {
  const getStageIcon = (stage: string) => {
    switch (stage) {
      case 'draft':
        return <FileText className="w-5 h-5" />;
      case 'review':
        return <Users className="w-5 h-5" />;
      case 'approval':
        return <Shield className="w-5 h-5" />;
      case 'execution':
        return <PenTool className="w-5 h-5" />;
      default:
        return <Clock className="w-5 h-5" />;
    }
  };

  const getStageTitle = (stage: string) => {
    switch (stage) {
      case 'draft':
        return 'Contract Draft';
      case 'review':
        return 'Owner Review';
      case 'approval':
        return 'Admin Approval';
      case 'execution':
        return 'Digital Signing';
      default:
        return stage;
    }
  };

  const getStageDescription = (stage: string) => {
    switch (stage) {
      case 'draft':
        return 'Contract created and ready for review';
      case 'review':
        return 'Property owner reviewing contract terms';
      case 'approval':
        return 'NAL admin reviewing for final approval';
      case 'execution':
        return 'Digital signatures from all parties';
      default:
        return 'Processing stage';
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed':
        return 'text-green-600 bg-green-100';
      case 'current':
        return 'text-blue-600 bg-blue-100';
      case 'pending':
        return 'text-gray-400 bg-gray-100';
      default:
        return 'text-gray-400 bg-gray-100';
    }
  };

  // Create workflow steps with status
  const workflowSteps = [
    { stage: 'draft', order: 1 },
    { stage: 'review', order: 2 },
    { stage: 'approval', order: 3 },
    { stage: 'execution', order: 4 }
  ].map(step => {
    const workflowItem = workflow.find(w => w.stage === step.stage);
    let status: 'pending' | 'completed' | 'current' = 'pending';
    
    if (workflowItem?.status === 'completed') {
      status = 'completed';
    } else if (step.stage === currentStage) {
      status = 'current';
    }

    return {
      ...step,
      status,
      completedAt: workflowItem?.completedAt,
      assignedTo: workflowItem?.assignedTo,
      comments: workflowItem?.comments
    };
  });

  return (
    <Card>
      <CardHeader>
        <CardTitle>Contract Workflow</CardTitle>
      </CardHeader>
      <CardContent>
        <div className="space-y-4">
          {workflowSteps.map((step, index) => (
            <div key={step.stage} className="flex items-start gap-4">
              {/* Step Icon */}
              <div className={`flex-shrink-0 w-10 h-10 rounded-full flex items-center justify-center ${getStatusColor(step.status)}`}>
                {step.status === 'completed' ? (
                  <CheckCircle className="w-5 h-5" />
                ) : (
                  getStageIcon(step.stage)
                )}
              </div>

              {/* Step Content */}
              <div className="flex-1 min-w-0">
                <div className="flex items-center gap-2 mb-1">
                  <h4 className="font-medium text-gray-900">
                    {getStageTitle(step.stage)}
                  </h4>
                  <Badge variant="outline" className={getStatusColor(step.status)}>
                    {step.status === 'current' ? 'In Progress' : step.status}
                  </Badge>
                </div>
                
                <p className="text-sm text-gray-600 mb-2">
                  {getStageDescription(step.stage)}
                </p>

                {step.completedAt && (
                  <p className="text-xs text-gray-500">
                    Completed on {new Date(step.completedAt).toLocaleDateString()}
                  </p>
                )}

                {step.comments && (
                  <div className="mt-2 p-2 bg-gray-50 rounded text-sm">
                    <span className="font-medium">Comments: </span>
                    {step.comments}
                  </div>
                )}
              </div>

              {/* Connector Line */}
              {index < workflowSteps.length - 1 && (
                <div className="absolute left-5 mt-10 w-0.5 h-8 bg-gray-200" />
              )}
            </div>
          ))}
        </div>
      </CardContent>
    </Card>
  );
};