import React, { useState } from 'react';
import { Card } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { FileText, Calculator, CheckCircle, Download } from 'lucide-react';

interface EStampCalculatorProps {
  onGenerate?: (certificateData: any) => void;
}

export const EStampCalculator: React.FC<EStampCalculatorProps> = ({ onGenerate }) => {
  const [formData, setFormData] = useState({
    state: '',
    documentType: '',
    propertyType: '',
    propertyValue: '',
  });
  
  const [calculatedDuty, setCalculatedDuty] = useState<number | null>(null);
  const [loading, setLoading] = useState(false);
  const [certificate, setCertificate] = useState<any>(null);

  const states = [
    'Karnataka',
    'Maharashtra',
    'Tamil Nadu',
    'Delhi',
    'Gujarat',
    'Rajasthan',
    'Uttar Pradesh',
    'West Bengal'
  ];

  const documentTypes = [
    { value: 'sale_deed', label: 'Sale Deed' },
    { value: 'rent_agreement', label: 'Rent Agreement' },
    { value: 'lease_deed', label: 'Lease Deed' },
    { value: 'power_of_attorney', label: 'Power of Attorney' },
    { value: 'gift_deed', label: 'Gift Deed' },
    { value: 'mortgage_deed', label: 'Mortgage Deed' }
  ];

  const propertyTypes = [
    { value: 'residential', label: 'Residential' },
    { value: 'commercial', label: 'Commercial' },
    { value: 'agricultural', label: 'Agricultural' },
    { value: 'industrial', label: 'Industrial' }
  ];

  const handleCalculate = async () => {
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/estamp/calculate', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('access_token') || localStorage.getItem('sellerToken')}`
        },
        body: JSON.stringify({
          state: formData.state,
          document_type: formData.documentType,
          property_type: formData.propertyType,
          property_value: parseFloat(formData.propertyValue)
        })
      });

      if (response.ok) {
        const data = await response.json();
        setCalculatedDuty(data.stamp_duty_amount);
      }
    } catch (error) {
      console.error('Error calculating stamp duty:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleGenerate = async () => {
    setLoading(true);
    try {
      const response = await fetch('http://localhost:8090/api/estamp/', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('access_token') || localStorage.getItem('sellerToken')}`
        },
        body: JSON.stringify({
          state: formData.state,
          document_type: formData.documentType,
          property_type: formData.propertyType,
          property_value: parseFloat(formData.propertyValue),
          stamp_duty_amount: calculatedDuty,
          payment_method: 'UPI'
        })
      });

      if (response.ok) {
        const data = await response.json();
        setCertificate(data);
        if (onGenerate) {
          onGenerate(data);
        }
      }
    } catch (error) {
      console.error('Error generating e-stamp:', error);
    } finally {
      setLoading(false);
    }
  };

  const isFormValid = formData.state && formData.documentType && formData.propertyType && formData.propertyValue;

  return (
    <div className="space-y-6">
      <Card className="p-6">
        <div className="flex items-center gap-3 mb-6">
          <FileText className="w-6 h-6 text-blue-600" />
          <h3 className="text-xl font-semibold">E-Stamp Calculator</h3>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <Label htmlFor="state">State</Label>
            <Select value={formData.state} onValueChange={(value) => setFormData({ ...formData, state: value })}>
              <SelectTrigger>
                <SelectValue placeholder="Select state" />
              </SelectTrigger>
              <SelectContent>
                {states.map((state) => (
                  <SelectItem key={state} value={state}>
                    {state}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="documentType">Document Type</Label>
            <Select value={formData.documentType} onValueChange={(value) => setFormData({ ...formData, documentType: value })}>
              <SelectTrigger>
                <SelectValue placeholder="Select document type" />
              </SelectTrigger>
              <SelectContent>
                {documentTypes.map((type) => (
                  <SelectItem key={type.value} value={type.value}>
                    {type.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="propertyType">Property Type</Label>
            <Select value={formData.propertyType} onValueChange={(value) => setFormData({ ...formData, propertyType: value })}>
              <SelectTrigger>
                <SelectValue placeholder="Select property type" />
              </SelectTrigger>
              <SelectContent>
                {propertyTypes.map((type) => (
                  <SelectItem key={type.value} value={type.value}>
                    {type.label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          <div>
            <Label htmlFor="propertyValue">Property Value (₹)</Label>
            <Input
              id="propertyValue"
              type="number"
              placeholder="Enter property value"
              value={formData.propertyValue}
              onChange={(e) => setFormData({ ...formData, propertyValue: e.target.value })}
            />
          </div>
        </div>

        <div className="mt-6 flex gap-3">
          <Button
            onClick={handleCalculate}
            disabled={!isFormValid || loading}
            className="flex items-center gap-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white"
          >
            <Calculator className="w-4 h-4" />
            Calculate Stamp Duty
          </Button>
        </div>

        {calculatedDuty !== null && (
          <div className="mt-6 p-4 bg-blue-50 border border-blue-200 rounded-lg">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Calculated Stamp Duty</p>
                <p className="text-2xl font-bold text-blue-600">₹{calculatedDuty.toLocaleString('en-IN')}</p>
              </div>
              <Button
                onClick={handleGenerate}
                disabled={loading}
                className="flex items-center gap-2 bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 text-white"
              >
                <CheckCircle className="w-4 h-4" />
                Generate E-Stamp
              </Button>
            </div>
          </div>
        )}
      </Card>

      {certificate && (
        <Card className="p-6 bg-green-50 border-green-200">
          <div className="flex items-center gap-3 mb-4">
            <CheckCircle className="w-6 h-6 text-green-600" />
            <h3 className="text-xl font-semibold text-green-800">E-Stamp Generated Successfully</h3>
          </div>
          
          <div className="space-y-2">
            <p className="text-sm">
              <span className="font-semibold">Certificate Number:</span> {certificate.certificate_number}
            </p>
            <p className="text-sm">
              <span className="font-semibold">Amount:</span> ₹{certificate.stamp_duty_amount}
            </p>
            <p className="text-sm">
              <span className="font-semibold">Status:</span> {certificate.payment_status}
            </p>
          </div>

          <Button className="mt-4 flex items-center gap-2" variant="outline">
            <Download className="w-4 h-4" />
            Download Certificate
          </Button>
        </Card>
      )}
    </div>
  );
};
