import React, { useState, useEffect } from 'react';
import { 
  Card, 
  CardContent, 
  CardHeader, 
  CardTitle 
} from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  Download, 
  FileText, 
  Shield, 
  CheckCircle, 
  AlertCircle,
  Calendar,
  MapPin,
  Hash,
  IndianRupee
} from 'lucide-react';
import { toast } from 'sonner';

const EStampCertificate = ({ contractId, contractData }) => {
  const [estampData, setEstampData] = useState(null);
  const [loading, setLoading] = useState(true);
  const [downloading, setDownloading] = useState(false);

  useEffect(() => {
    if (contractId) {
      fetchEStampData();
    }
  }, [contractId]);

  const fetchEStampData = async () => {
    try {
      setLoading(true);
      const response = await fetch(`/api/estamp/contract/${contractId}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });

      if (response.ok) {
        const result = await response.json();
        setEstampData(result.data);
      } else if (response.status === 404) {
        setEstampData(null);
      } else {
        throw new Error('Failed to fetch e-stamp data');
      }
    } catch (error) {
      console.error('Error fetching e-stamp:', error);
      toast.error('Failed to load e-stamp certificate');
    } finally {
      setLoading(false);
    }
  };

  const downloadEStampPDF = async () => {
    if (!estampData) return;

    try {
      setDownloading(true);
      const response = await fetch(`/api/estamp/download/${estampData.id}`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });

      if (!response.ok) {
        throw new Error('Failed to download e-stamp certificate');
      }

      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const link = document.createElement('a');
      link.href = url;
      link.download = `estamp_${estampData.certificate_number}.pdf`;
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      window.URL.revokeObjectURL(url);

      toast.success('E-stamp certificate downloaded successfully');
    } catch (error) {
      console.error('Error downloading e-stamp:', error);
      toast.error('Failed to download e-stamp certificate');
    } finally {
      setDownloading(false);
    }
  };

  const generateEStamp = async () => {
    try {
      setLoading(true);
      
      const calcResponse = await fetch('/api/estamp/calculate-duty', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          state: contractData?.property?.state || 'Karnataka',
          document_type: 'sale_deed',
          property_type: 'residential',
          property_value: contractData?.agreed_price || 1000000
        })
      });

      if (!calcResponse.ok) {
        throw new Error('Failed to calculate stamp duty');
      }

      const createResponse = await fetch('/api/estamp/create', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          state: contractData?.property?.state || 'Karnataka',
          document_type: 'sale_deed',
          property_value: contractData?.agreed_price || 1000000,
          payment_method: 'UPI',
          contract_id: contractId,
          property_id: contractData?.property_id
        })
      });

      if (!createResponse.ok) {
        throw new Error('Failed to create e-stamp certificate');
      }

      const createResult = await createResponse.json();

      const paymentResponse = await fetch('/api/estamp/process-payment', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        },
        body: JSON.stringify({
          estamp_id: createResult.data.id,
          payment_gateway: 'UPI',
          payment_method: 'UPI',
          transaction_id: `TXN${Date.now()}`
        })
      });

      if (!paymentResponse.ok) {
        throw new Error('Failed to process payment');
      }

      toast.success('E-stamp certificate generated successfully');
      fetchEStampData();
    } catch (error) {
      console.error('Error generating e-stamp:', error);
      toast.error('Failed to generate e-stamp certificate');
    } finally {
      setLoading(false);
    }
  };

  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    return new Date(dateString).toLocaleDateString('en-IN', {
      day: '2-digit',
      month: 'short',
      year: 'numeric'
    });
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0
    }).format(amount);
  };

  if (loading) {
    return (
      <Card>
        <CardContent className="p-6">
          <div className="flex items-center justify-center h-32">
            <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
          </div>
        </CardContent>
      </Card>
    );
  }

  if (!estampData) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <FileText className="h-5 w-5" />
            E-Stamp Certificate
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="text-center py-8">
            <AlertCircle className="h-12 w-12 text-amber-500 mx-auto mb-4" />
            <h3 className="text-lg font-semibold mb-2">No E-Stamp Certificate</h3>
            <p className="text-gray-600 mb-4">
              This contract doesn't have an e-stamp certificate yet.
            </p>
            {contractData?.status === 'signed' && (
              <Button onClick={generateEStamp} disabled={loading}>
                Generate E-Stamp Certificate
              </Button>
            )}
          </div>
        </CardContent>
      </Card>
    );
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Shield className="h-5 w-5 text-green-600" />
            E-Stamp Certificate
          </div>
          <Badge variant="success" className="flex items-center gap-1">
            <CheckCircle className="h-3 w-3" />
            Verified
          </Badge>
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <div className="flex items-center gap-2 mb-2">
            <Hash className="h-4 w-4 text-blue-600" />
            <span className="text-sm font-medium text-blue-800">Certificate Number</span>
          </div>
          <p className="text-lg font-bold text-blue-900 font-mono">
            {estampData.certificate_number}
          </p>
        </div>

        <div className="bg-green-50 border border-green-200 rounded-lg p-4">
          <div className="flex items-center gap-2 mb-2">
            <IndianRupee className="h-4 w-4 text-green-600" />
            <span className="text-sm font-medium text-green-800">Stamp Duty Paid</span>
          </div>
          <p className="text-2xl font-bold text-green-900">
            {formatCurrency(estampData.stamp_duty_amount)}
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="space-y-3">
            <div>
              <label className="text-sm font-medium text-gray-600">State</label>
              <p className="text-sm text-gray-900">{estampData.state}</p>
            </div>
            <div>
              <label className="text-sm font-medium text-gray-600">Document Type</label>
              <p className="text-sm text-gray-900 capitalize">
                {estampData.document_type.replace('_', ' ')}
              </p>
            </div>
            <div>
              <label className="text-sm font-medium text-gray-600">Issued By</label>
              <p className="text-sm text-gray-900">{estampData.issued_by}</p>
            </div>
          </div>
          
          <div className="space-y-3">
            <div>
              <label className="text-sm font-medium text-gray-600">Issue Date</label>
              <div className="flex items-center gap-1">
                <Calendar className="h-3 w-3 text-gray-500" />
                <p className="text-sm text-gray-900">{formatDate(estampData.issued_at)}</p>
              </div>
            </div>
            <div>
              <label className="text-sm font-medium text-gray-600">Valid Until</label>
              <div className="flex items-center gap-1">
                <Calendar className="h-3 w-3 text-gray-500" />
                <p className="text-sm text-gray-900">{formatDate(estampData.valid_until)}</p>
              </div>
            </div>
            <div>
              <label className="text-sm font-medium text-gray-600">Payment Status</label>
              <Badge variant="success" className="text-xs">
                {estampData.payment_status}
              </Badge>
            </div>
          </div>
        </div>

        <div className="border-t pt-4">
          <label className="text-sm font-medium text-gray-600">Security Hash</label>
          <p className="text-xs text-gray-500 font-mono break-all mt-1">
            {estampData.certificate_hash}
          </p>
        </div>

        <div className="border-t pt-4">
          <Button 
            onClick={downloadEStampPDF} 
            disabled={downloading}
            className="w-full"
          >
            <Download className="h-4 w-4 mr-2" />
            {downloading ? 'Downloading...' : 'Download E-Stamp Certificate'}
          </Button>
        </div>

        <div className="bg-gray-50 border border-gray-200 rounded-lg p-3">
          <p className="text-xs text-gray-600">
            <strong>Legal Notice:</strong> This is a digitally generated e-stamp certificate 
            issued by Stock Holding Corporation of India Limited (SHCIL). It is legally 
            valid for the specified document type.
          </p>
        </div>
      </CardContent>
    </Card>
  );
};

export default EStampCertificate;