import React, { useState, useEffect } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Download, Eye, FileText, Shield, Calendar, MapPin, User, Building, CheckCircle, AlertCircle } from 'lucide-react';
import { toast } from '@/utils/toast';

interface StampPaperViewerProps {
  isOpen: boolean;
  onClose: () => void;
  contractId: string;
}

interface StampPaperData {
  certificate_number: string;
  stamp_duty_amount: number;
  generated_at: string;
  is_valid: boolean;
  contract_details: {
    contract_number: string;
    contract_amount: number;
    property_id: number;
    contract_title?: string;
    contract_type?: string;
    start_date?: string;
    end_date?: string;
    buyer_name?: string;
    seller_name?: string;
    buyer_email?: string;
    seller_email?: string;
    property_address?: string;
  };
  signatures: Array<{
    signer_name: string;
    signer_role: string;
    signed_at: string;
    email: string;
  }>;
}

export const StampPaperViewer: React.FC<StampPaperViewerProps> = ({
  isOpen,
  onClose,
  contractId
}) => {
  const [stampData, setStampData] = useState<StampPaperData | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    if (isOpen && contractId) {
      loadStampPaperData();
    }
  }, [isOpen, contractId]);

  const loadStampPaperData = async () => {
    try {
      setLoading(true);
      setError(null);

      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/complete-details`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (!response.ok) {
        throw new Error('Failed to load stamp paper details');
      }

      const data = await response.json();
      
      if (!data.stamp_paper) {
        // If no stamp paper exists, try to generate one or show contract details anyway
        console.log('No stamp paper found, showing contract details with placeholder stamp paper');
      }

      // Format the data for display
      const formattedData: StampPaperData = {
        certificate_number: data.stamp_paper?.certificate_number || `CERT-${contractId}-${Date.now()}`,
        stamp_duty_amount: data.stamp_paper?.stamp_duty_amount || 100,
        generated_at: data.stamp_paper?.generated_at || new Date().toISOString(),
        is_valid: data.stamp_paper?.is_valid ?? true,
        contract_details: {
          contract_number: data.contract.contract_number || `CONTRACT-${contractId}`,
          contract_amount: data.contract.contract_amount || 0,
          property_id: data.contract.property_id,
          contract_title: data.contract.metadata?.title || 'Property Sale Agreement',
          contract_type: data.contract.contract_type || 'sale',
          start_date: data.contract.metadata?.start_date || data.contract.created_at,
          end_date: data.contract.metadata?.end_date,
          buyer_name: data.contract.buyer_email?.split('@')[0] || 'Buyer',
          seller_name: data.contract.seller_email?.split('@')[0] || 'Seller',
          buyer_email: data.contract.buyer_email,
          seller_email: data.contract.seller_email,
          property_address: data.contract.property_address || data.contract.property_title || 'Property Address'
        },
        signatures: (data.signatures || []).map((sig: any) => ({
          signer_name: sig.email?.split('@')[0] || 'User',
          signer_role: sig.role,
          signed_at: sig.signed_at,
          email: sig.email
        }))
      };

      setStampData(formattedData);
    } catch (err: any) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  const handleDownload = async () => {
    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `stamp_paper_${stampData?.certificate_number || contractId}.pdf`;
        document.body.appendChild(a);
        a.click();
        window.URL.revokeObjectURL(url);
        document.body.removeChild(a);
        toast.success('Stamp paper downloaded successfully!');
      } else {
        throw new Error('Failed to download stamp paper');
      }
    } catch (error) {
      toast.error('Failed to download stamp paper');
    }
  };

  const handleGenerateStampPaper = async () => {
    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper/generate`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        toast.success('Stamp paper generated successfully!');
        // Reload the data
        loadStampPaperData();
      } else {
        throw new Error('Failed to generate stamp paper');
      }
    } catch (error) {
      toast.error('Failed to generate stamp paper');
    }
  };

  const handleViewPDF = async () => {
    try {
      const token = localStorage.getItem('access_token') || localStorage.getItem('sellerToken');
      const response = await fetch(`http://localhost:8090/api/contracts/${contractId}/stamp-paper/view`, {
        headers: {
          'Authorization': `Bearer ${token}`
        }
      });

      if (response.ok) {
        const blob = await response.blob();
        const url = window.URL.createObjectURL(blob);
        window.open(url, '_blank');
        setTimeout(() => window.URL.revokeObjectURL(url), 1000);
      } else {
        throw new Error('Failed to view stamp paper');
      }
    } catch (error) {
      toast.error('Failed to view stamp paper');
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-IN', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR'
    }).format(amount);
  };

  if (loading) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-5xl max-h-[95vh] overflow-y-auto">
          <div className="text-center py-8">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading stamp paper details...</p>
          </div>
        </DialogContent>
      </Dialog>
    );
  }

  if (error) {
    return (
      <Dialog open={isOpen} onOpenChange={onClose}>
        <DialogContent className="max-w-5xl max-h-[95vh] overflow-y-auto">
          <div className="text-center py-8">
            <AlertCircle className="w-16 h-16 text-red-500 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-red-900 mb-2">Error</h3>
            <p className="text-red-700 mb-4">{error}</p>
            <Button onClick={onClose}>Close</Button>
          </div>
        </DialogContent>
      </Dialog>
    );
  }

  return (
    <Dialog open={isOpen} onOpenChange={onClose}>
      <DialogContent className="max-w-5xl max-h-[95vh] overflow-y-auto">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <FileText className="w-6 h-6 text-blue-600" />
            Digital Stamp Paper - {stampData?.certificate_number}
          </DialogTitle>
        </DialogHeader>

        {stampData && (
          <div className="space-y-6 max-h-[80vh] overflow-y-auto pr-2">
            {/* Status Banner */}
            <div className={`p-4 rounded-lg border ${
              stampData.certificate_number.startsWith('CERT-') 
                ? 'bg-yellow-50 border-yellow-200' 
                : stampData.is_valid 
                  ? 'bg-green-50 border-green-200' 
                  : 'bg-red-50 border-red-200'
            }`}>
              <div className="flex items-center gap-2">
                {stampData.certificate_number.startsWith('CERT-') ? (
                  <AlertCircle className="w-5 h-5 text-yellow-600" />
                ) : stampData.is_valid ? (
                  <CheckCircle className="w-5 h-5 text-green-600" />
                ) : (
                  <AlertCircle className="w-5 h-5 text-red-600" />
                )}
                <span className={`font-semibold ${
                  stampData.certificate_number.startsWith('CERT-') 
                    ? 'text-yellow-900' 
                    : stampData.is_valid 
                      ? 'text-green-900' 
                      : 'text-red-900'
                }`}>
                  {stampData.certificate_number.startsWith('CERT-') 
                    ? 'Stamp Paper Not Generated Yet' 
                    : stampData.is_valid 
                      ? 'Valid Digital Stamp Paper' 
                      : 'Invalid Stamp Paper'
                  }
                </span>
              </div>
              {stampData.certificate_number.startsWith('CERT-') && (
                <p className="text-yellow-800 text-sm mt-1">
                  Click "Generate Stamp Paper" to create the official e-stamp certificate.
                </p>
              )}
            </div>

            {/* Certificate Details */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Shield className="w-5 h-5 text-blue-600" />
                  Certificate Information
                </CardTitle>
              </CardHeader>
              <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="text-sm font-medium text-gray-600">Certificate Number</label>
                  <p className="text-lg font-mono bg-gray-100 p-2 rounded">{stampData.certificate_number}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Stamp Duty Amount</label>
                  <p className="text-lg font-semibold text-green-600">{formatCurrency(stampData.stamp_duty_amount)}</p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Generated On</label>
                  <p className="text-sm flex items-center gap-1">
                    <Calendar className="w-4 h-4" />
                    {formatDate(stampData.generated_at)}
                  </p>
                </div>
                <div>
                  <label className="text-sm font-medium text-gray-600">Status</label>
                  <Badge variant={stampData.is_valid ? "default" : "destructive"}>
                    {stampData.is_valid ? 'Valid' : 'Invalid'}
                  </Badge>
                </div>
              </CardContent>
            </Card>

            {/* Contract Details */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Building className="w-5 h-5 text-blue-600" />
                  Contract Information
                </CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <label className="text-sm font-medium text-gray-600">Contract Title</label>
                    <p className="font-semibold">{stampData.contract_details.contract_title || 'Property Sale Agreement'}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Contract Type</label>
                    <p className="capitalize">{stampData.contract_details.contract_type || 'Sale'}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Contract Number</label>
                    <p className="font-mono">{stampData.contract_details.contract_number}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Contract Amount</label>
                    <p className="text-lg font-semibold text-green-600">{formatCurrency(stampData.contract_details.contract_amount)}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Property ID</label>
                    <p>#{stampData.contract_details.property_id}</p>
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Start Date</label>
                    <p className="flex items-center gap-1">
                      <Calendar className="w-4 h-4" />
                      {stampData.contract_details.start_date ? formatDate(stampData.contract_details.start_date) : 'Not specified'}
                    </p>
                  </div>
                  {stampData.contract_details.end_date && (
                    <div>
                      <label className="text-sm font-medium text-gray-600">End Date</label>
                      <p className="flex items-center gap-1">
                        <Calendar className="w-4 h-4" />
                        {formatDate(stampData.contract_details.end_date)}
                      </p>
                    </div>
                  )}
                </div>
                
                <div>
                  <label className="text-sm font-medium text-gray-600">Property Address</label>
                  <p className="flex items-start gap-1 mt-1">
                    <MapPin className="w-4 h-4 mt-0.5 text-blue-500" />
                    <span>{stampData.contract_details.property_address}</span>
                  </p>
                </div>
                
                {/* Parties Information */}
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4 pt-4 border-t">
                  <div>
                    <label className="text-sm font-medium text-gray-600">Buyer</label>
                    <p className="font-medium">{stampData.contract_details.buyer_name}</p>
                    {stampData.contract_details.buyer_email && (
                      <p className="text-sm text-gray-500">{stampData.contract_details.buyer_email}</p>
                    )}
                  </div>
                  <div>
                    <label className="text-sm font-medium text-gray-600">Seller</label>
                    <p className="font-medium">{stampData.contract_details.seller_name}</p>
                    {stampData.contract_details.seller_email && (
                      <p className="text-sm text-gray-500">{stampData.contract_details.seller_email}</p>
                    )}
                  </div>
                </div>
              </CardContent>
            </Card>

            {/* Signatures */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <User className="w-5 h-5 text-blue-600" />
                  Digital Signatures ({stampData.signatures.length})
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-3">
                  {stampData.signatures.map((signature, index) => (
                    <div key={index} className="flex items-center justify-between p-3 bg-green-50 border border-green-200 rounded-lg">
                      <div className="flex items-center gap-3">
                        <CheckCircle className="w-5 h-5 text-green-600" />
                        <div>
                          <p className="font-medium">{signature.signer_name}</p>
                          <p className="text-sm text-gray-600">{signature.email}</p>
                        </div>
                      </div>
                      <div className="text-right">
                        <Badge variant="outline" className="mb-1">
                          {signature.signer_role.toUpperCase()}
                        </Badge>
                        <p className="text-xs text-gray-500">
                          Signed: {formatDate(signature.signed_at)}
                        </p>
                      </div>
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>

            {/* Action Buttons */}
            <div className="flex gap-3 justify-end pt-4 border-t">
              <Button variant="outline" onClick={onClose}>
                Close
              </Button>
              
              {/* Show generate button if no stamp paper exists */}
              {!stampData?.certificate_number.startsWith('CERT-') && (
                <Button onClick={handleGenerateStampPaper} className="flex items-center gap-2 bg-orange-600 hover:bg-orange-700">
                  <FileText className="w-4 h-4" />
                  Generate Stamp Paper
                </Button>
              )}
              
              <Button onClick={handleViewPDF} className="flex items-center gap-2">
                <Eye className="w-4 h-4" />
                View PDF
              </Button>
              <Button onClick={handleDownload} className="flex items-center gap-2 bg-green-600 hover:bg-green-700">
                <Download className="w-4 h-4" />
                Download Stamp Paper
              </Button>
            </div>

            {/* Legal Notice */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 text-sm">
              <p className="font-medium text-blue-900 mb-2">Legal Notice:</p>
              <p className="text-blue-800">
                This is a digitally generated e-stamp certificate issued by the Government of India. 
                It has the same legal validity as a physical stamp paper. Any tampering with this 
                document will render it invalid. For verification, please visit the official 
                e-stamp verification portal.
              </p>
            </div>
          </div>
        )}
      </DialogContent>
    </Dialog>
  );
};