import React, { useState } from 'react';
import { 
  FaUserShield, FaCheck, FaTimes, FaEye, FaDownload, 
  FaSpinner, FaSearch, FaFilter, FaClock, FaExclamationTriangle
} from 'react-icons/fa';
import { useErasureRequests, useDataExports } from '../../hooks/useDataRetention';

interface AdminDataRetentionProps {
  isAdmin: boolean;
}

const AdminDataRetention: React.FC<AdminDataRetentionProps> = ({ isAdmin }) => {
  const [activeTab, setActiveTab] = useState('requests');
  const [statusFilter, setStatusFilter] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');
  const { requests, updateRequest, loading } = useErasureRequests();
  const { exports } = useDataExports();
  const [processing, setProcessing] = useState<string | null>(null);

  if (!isAdmin) {
    return (
      <div className="admin-access-denied">
        <FaUserShield />
        <h3>Admin Access Required</h3>
        <p>You need administrator privileges to access this section.</p>
      </div>
    );
  }

  const handleRequestAction = async (requestId: number, action: 'approve' | 'reject', notes?: string) => {
    try {
      setProcessing(requestId.toString());
      await updateRequest(requestId, {
        status: action === 'approve' ? 'approved' : 'rejected',
        notes: notes || `Request ${action}d by admin`
      });
      alert(`Request ${action}d successfully!`);
    } catch (error) {
      alert(`Failed to ${action} request. Please try again.`);
    } finally {
      setProcessing(null);
    }
  };

  const filteredRequests = requests?.filter(request => {
    const matchesStatus = statusFilter === 'all' || request.status === statusFilter;
    const matchesSearch = !searchTerm || 
      request.user_id.toLowerCase().includes(searchTerm.toLowerCase()) ||
      request.reason.toLowerCase().includes(searchTerm.toLowerCase());
    return matchesStatus && matchesSearch;
  }) || [];

  const getStatusBadge = (status: string) => {
    const statusClasses = {
      pending: 'status-pending',
      approved: 'status-approved', 
      rejected: 'status-rejected',
      completed: 'status-completed'
    };
    return <span className={`status-badge ${statusClasses[status as keyof typeof statusClasses]}`}>{status}</span>;
  };

  return (
    <div className="admin-data-retention">
      <div className="admin-header">
        <h2><FaUserShield /> Admin: Data Retention Management</h2>
        <p>Review and manage user data requests and retention policies</p>
      </div>

      <div className="admin-tabs">
        <button 
          className={`tab ${activeTab === 'requests' ? 'active' : ''}`}
          onClick={() => setActiveTab('requests')}
        >
          Erasure Requests ({requests?.filter(r => r.status === 'pending').length || 0})
        </button>
        <button 
          className={`tab ${activeTab === 'exports' ? 'active' : ''}`}
          onClick={() => setActiveTab('exports')}
        >
          Data Exports
        </button>
      </div>

      {activeTab === 'requests' && (
        <div className="requests-section">
          <div className="section-controls">
            <div className="search-bar">
              <FaSearch />
              <input
                type="text"
                placeholder="Search by user ID or reason..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
              />
            </div>
            
            <div className="filter-bar">
              <FaFilter />
              <select value={statusFilter} onChange={(e) => setStatusFilter(e.target.value)}>
                <option value="all">All Status</option>
                <option value="pending">Pending</option>
                <option value="approved">Approved</option>
                <option value="rejected">Rejected</option>
                <option value="completed">Completed</option>
              </select>
            </div>
          </div>

          {loading ? (
            <div className="loading-spinner">
              <FaSpinner className="spinning" />
              <p>Loading erasure requests...</p>
            </div>
          ) : (
            <div className="requests-table">
              <table>
                <thead>
                  <tr>
                    <th>Request ID</th>
                    <th>User ID</th>
                    <th>Type</th>
                    <th>Status</th>
                    <th>Requested</th>
                    <th>Data Categories</th>
                    <th>Actions</th>
                  </tr>
                </thead>
                <tbody>
                  {filteredRequests.map((request) => (
                    <tr key={request.id}>
                      <td>#{request.id}</td>
                      <td className="user-id">{request.user_id.substring(0, 8)}...</td>
                      <td>
                        <span className={`type-badge ${request.request_type}`}>
                          {request.request_type}
                        </span>
                      </td>
                      <td>{getStatusBadge(request.status)}</td>
                      <td>
                        <FaClock />
                        {new Date(request.requested_at).toLocaleDateString()}
                      </td>
                      <td>
                        {request.data_categories.length > 0 ? (
                          <div className="categories-list">
                            {request.data_categories.slice(0, 2).map(cat => (
                              <span key={cat} className="category-tag">{cat}</span>
                            ))}
                            {request.data_categories.length > 2 && (
                              <span className="category-more">+{request.data_categories.length - 2}</span>
                            )}
                          </div>
                        ) : (
                          <span className="no-categories">All data</span>
                        )}
                      </td>
                      <td>
                        <div className="action-buttons">
                          {request.status === 'pending' && (
                            <>
                              <button
                                className="btn btn-success btn-sm"
                                onClick={() => handleRequestAction(request.id, 'approve')}
                                disabled={processing === request.id.toString()}
                                title="Approve Request"
                              >
                                {processing === request.id.toString() ? 
                                  <FaSpinner className="spinning" /> : <FaCheck />
                                }
                              </button>
                              <button
                                className="btn btn-danger btn-sm"
                                onClick={() => handleRequestAction(request.id, 'reject')}
                                disabled={processing === request.id.toString()}
                                title="Reject Request"
                              >
                                <FaTimes />
                              </button>
                            </>
                          )}
                          <button
                            className="btn btn-info btn-sm"
                            onClick={() => {/* Show details modal */}}
                            title="View Details"
                          >
                            <FaEye />
                          </button>
                        </div>
                      </td>
                    </tr>
                  ))}
                </tbody>
              </table>
              
              {filteredRequests.length === 0 && (
                <div className="no-data">
                  <p>No erasure requests found matching your criteria.</p>
                </div>
              )}
            </div>
          )}
        </div>
      )}

      {activeTab === 'exports' && (
        <div className="exports-section">
          <div className="exports-table">
            <table>
              <thead>
                <tr>
                  <th>Export ID</th>
                  <th>User ID</th>
                  <th>Type</th>
                  <th>Status</th>
                  <th>Requested</th>
                  <th>File Size</th>
                  <th>Actions</th>
                </tr>
              </thead>
              <tbody>
                {exports?.map((exportItem) => (
                  <tr key={exportItem.id}>
                    <td>#{exportItem.id}</td>
                    <td className="user-id">{exportItem.user_id.substring(0, 8)}...</td>
                    <td>
                      <span className="export-type">{exportItem.export_type}</span>
                    </td>
                    <td>{getStatusBadge(exportItem.status)}</td>
                    <td>
                      <FaClock />
                      {new Date(exportItem.requested_at).toLocaleDateString()}
                    </td>
                    <td>
                      {exportItem.file_size ? 
                        `${(exportItem.file_size / 1024 / 1024).toFixed(2)} MB` : 
                        'N/A'
                      }
                    </td>
                    <td>
                      <div className="action-buttons">
                        {exportItem.status === 'completed' && exportItem.file_path && (
                          <button
                            className="btn btn-primary btn-sm"
                            onClick={() => window.open(`/api/data-retention/exports/${exportItem.id}/download`)}
                            title="Download Export"
                          >
                            <FaDownload />
                          </button>
                        )}
                        <button
                          className="btn btn-info btn-sm"
                          onClick={() => {/* Show details */}}
                          title="View Details"
                        >
                          <FaEye />
                        </button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      <div className="admin-guidelines">
        <h3><FaExclamationTriangle /> Admin Guidelines</h3>
        <div className="guidelines-grid">
          <div className="guideline-item">
            <h4>Erasure Request Review</h4>
            <ul>
              <li>Review each request within 30 days</li>
              <li>Verify user identity before approval</li>
              <li>Consider legal obligations before erasure</li>
              <li>Document reasons for rejection</li>
            </ul>
          </div>
          
          <div className="guideline-item">
            <h4>Data Export Management</h4>
            <ul>
              <li>Exports expire after 7 days</li>
              <li>Monitor file sizes and storage usage</li>
              <li>Ensure data accuracy in exports</li>
              <li>Log all download activities</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AdminDataRetention;