import React, { useState } from 'react';
import { 
  FaDownload, FaSpinner, FaFileExport, FaCheck, FaTimes, 
  FaClock, FaExclamationTriangle, FaCalendarAlt
} from 'react-icons/fa';
import { useDataExports } from '../../hooks/useDataRetention';
import { DataExport } from '../../types/dataRetention';

const DataExports: React.FC = () => {
  const { exports, loading, error, requestExport, downloadExport } = useDataExports();
  const [requesting, setRequesting] = useState(false);
  const [downloading, setDownloading] = useState<number | null>(null);

  const handleRequestExport = async (exportType: 'personal_data' | 'activity_logs' | 'property_data' | 'complete') => {
    try {
      setRequesting(true);
      const newExport = await requestExport(exportType);
      
      // Show appropriate message based on export status
      if (newExport.status === 'completed') {
        alert('Export completed! Download started automatically.');
      } else {
        alert('Export request submitted! You will be notified when it\'s ready.');
      }
    } catch (err) {
      console.error('Failed to request export:', err);
      alert('Failed to request export. Please try again.');
    } finally {
      setRequesting(false);
    }
  };

  const handleDownload = async (exportItem: DataExport) => {
    try {
      setDownloading(exportItem.id);
      await downloadExport(exportItem.id, `data_export_${exportItem.export_type}_${exportItem.id}.zip`);
    } catch (err) {
      console.error('Failed to download export:', err);
    } finally {
      setDownloading(null);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <FaClock className="status-icon pending" />;
      case 'processing': return <FaSpinner className="status-icon processing spinning" />;
      case 'completed': return <FaCheck className="status-icon completed" />;
      case 'failed': return <FaTimes className="status-icon failed" />;
      default: return <FaExclamationTriangle className="status-icon unknown" />;
    }
  };

  const getStatusClass = (status: string) => {
    return `status-badge ${status}`;
  };

  const formatFileSize = (bytes?: number) => {
    if (!bytes) return 'Unknown';
    const mb = bytes / (1024 * 1024);
    return `${mb.toFixed(2)} MB`;
  };

  const isDownloadExpired = (expiresAt?: string) => {
    if (!expiresAt) return false;
    return new Date(expiresAt) < new Date();
  };

  const getExportTypeLabel = (type: string) => {
    switch (type) {
      case 'personal_data': return 'Personal Data';
      case 'activity_logs': return 'Activity Logs';
      case 'property_data': return 'Property Data';
      case 'complete': return 'Complete Export';
      default: return type;
    }
  };

  const getExportTypeDescription = (type: string) => {
    switch (type) {
      case 'personal_data': return 'Your profile information, contact details, and account data';
      case 'activity_logs': return 'Your browsing history, search queries, and platform interactions';
      case 'property_data': return 'Saved properties, visit schedules, and property-related data';
      case 'complete': return 'All your data including profile, activity, and property information';
      default: return 'Data export';
    }
  };

  if (loading) {
    return (
      <div className="data-exports">
        <div className="loading-spinner">
          <FaSpinner className="spinning" />
          <p>Loading data exports...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="data-exports">
        <div className="error-message">
          <p>Error loading exports: {error}</p>
        </div>
      </div>
    );
  }

  return (
    <div className="data-exports">
      <div className="exports-header">
        <h2>Data Exports</h2>
        <p>Request and download your personal data for portability and backup purposes.</p>
      </div>

      <div className="export-options">
        <h3>Request New Export</h3>
        <div className="export-types-grid">
          <div className="export-type-card">
            <div className="export-type-icon">
              <FaFileExport />
            </div>
            <h4>Personal Data</h4>
            <p>{getExportTypeDescription('personal_data')}</p>
            <button
              className="btn btn-primary"
              onClick={() => handleRequestExport('personal_data')}
              disabled={requesting}
            >
              {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
              Request Export
            </button>
          </div>

          <div className="export-type-card">
            <div className="export-type-icon">
              <FaFileExport />
            </div>
            <h4>Activity Logs</h4>
            <p>{getExportTypeDescription('activity_logs')}</p>
            <button
              className="btn btn-primary"
              onClick={() => handleRequestExport('activity_logs')}
              disabled={requesting}
            >
              {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
              Request Export
            </button>
          </div>

          <div className="export-type-card">
            <div className="export-type-icon">
              <FaFileExport />
            </div>
            <h4>Property Data</h4>
            <p>{getExportTypeDescription('property_data')}</p>
            <button
              className="btn btn-primary"
              onClick={() => handleRequestExport('property_data')}
              disabled={requesting}
            >
              {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
              Request Export
            </button>
          </div>

          <div className="export-type-card">
            <div className="export-type-icon">
              <FaFileExport />
            </div>
            <h4>Complete Export</h4>
            <p>{getExportTypeDescription('complete')}</p>
            <button
              className="btn btn-primary"
              onClick={() => handleRequestExport('complete')}
              disabled={requesting}
            >
              {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
              Request Export
            </button>
          </div>
        </div>
      </div>

      <div className="exports-history">
        <h3>Export History</h3>
        {exports.length === 0 ? (
          <div className="empty-state">
            <p>No data exports found. Request your first export above.</p>
          </div>
        ) : (
          <div className="exports-list">
            {exports.map((exportItem) => (
              <div key={exportItem.id} className="export-card">
                <div className="export-header">
                  <div className="export-info">
                    <h4>{getExportTypeLabel(exportItem.export_type)}</h4>
                    <p className="export-description">
                      {getExportTypeDescription(exportItem.export_type)}
                    </p>
                  </div>
                  <div className={getStatusClass(exportItem.status)}>
                    {getStatusIcon(exportItem.status)}
                    {exportItem.status.toUpperCase()}
                  </div>
                </div>

                <div className="export-details">
                  <div className="detail-item">
                    <span className="label">Requested:</span>
                    <span className="value">
                      <FaCalendarAlt />
                      {new Date(exportItem.requested_at).toLocaleString()}
                    </span>
                  </div>

                  {exportItem.completed_at && (
                    <div className="detail-item">
                      <span className="label">Completed:</span>
                      <span className="value">
                        <FaCalendarAlt />
                        {new Date(exportItem.completed_at).toLocaleString()}
                      </span>
                    </div>
                  )}

                  {exportItem.file_size && (
                    <div className="detail-item">
                      <span className="label">File Size:</span>
                      <span className="value">{formatFileSize(exportItem.file_size)}</span>
                    </div>
                  )}

                  {exportItem.download_expires_at && (
                    <div className="detail-item">
                      <span className="label">Download Expires:</span>
                      <span className={`value ${isDownloadExpired(exportItem.download_expires_at) ? 'expired' : ''}`}>
                        <FaCalendarAlt />
                        {new Date(exportItem.download_expires_at).toLocaleString()}
                        {isDownloadExpired(exportItem.download_expires_at) && (
                          <span className="expired-label"> (Expired)</span>
                        )}
                      </span>
                    </div>
                  )}
                </div>

                <div className="export-actions">
                  {exportItem.status === 'completed' && !isDownloadExpired(exportItem.download_expires_at) && (
                    <button
                      className="btn btn-success"
                      onClick={() => handleDownload(exportItem)}
                      disabled={downloading === exportItem.id}
                    >
                      {downloading === exportItem.id ? (
                        <FaSpinner className="spinning" />
                      ) : (
                        <FaDownload />
                      )}
                      Download
                    </button>
                  )}

                  {exportItem.status === 'completed' && isDownloadExpired(exportItem.download_expires_at) && (
                    <button
                      className="btn btn-primary"
                      onClick={() => handleRequestExport(exportItem.export_type as any)}
                      disabled={requesting}
                    >
                      {requesting ? <FaSpinner className="spinning" /> : <FaFileExport />}
                      Request New Export
                    </button>
                  )}

                  {exportItem.status === 'failed' && (
                    <button
                      className="btn btn-primary"
                      onClick={() => handleRequestExport(exportItem.export_type as any)}
                      disabled={requesting}
                    >
                      {requesting ? <FaSpinner className="spinning" /> : <FaFileExport />}
                      Retry Export
                    </button>
                  )}
                </div>
              </div>
            ))}
          </div>
        )}
      </div>

      <div className="export-info-section">
        <h3>About Data Exports</h3>
        <div className="info-cards">
          <div className="info-card">
            <h4>Data Portability</h4>
            <p>
              Your data exports are provided in JSON format for easy portability. 
              You can use this data to transfer to another service or keep for your records.
            </p>
          </div>
          
          <div className="info-card">
            <h4>Security & Privacy</h4>
            <p>
              All exports are encrypted and password-protected. Download links expire after 7 days 
              for security purposes. Only you can access your data exports.
            </p>
          </div>
          
          <div className="info-card">
            <h4>Processing Time</h4>
            <p>
              Most exports are processed within a few minutes. Larger exports may take longer. 
              You'll be notified when your export is ready for download.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default DataExports;