import React, { useState } from 'react';
import { 
  FaShieldAlt, FaTrashAlt, FaDownload, FaUser, FaCog, FaChartBar,
  FaDatabase, FaCalendarAlt, FaFileExport, FaUserShield, FaSpinner
} from 'react-icons/fa';
import { useRetentionStats } from '../../hooks/useDataRetention';
import RetentionPolicies from './RetentionPolicies';
import ErasureRequests from './ErasureRequests';
import DataExports from './DataExports';
import UserDataSummary from './UserDataSummary';
import './DataRetentionDashboard.css';

const DataRetentionDashboard: React.FC = () => {
  const [activeTab, setActiveTab] = useState('overview');
  const { stats, loading: statsLoading } = useRetentionStats();

  if (statsLoading) {
    return (
      <div className="data-retention-dashboard">
        <div className="loading-spinner">
          <FaSpinner className="spinning" />
          <p>Loading data retention dashboard...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="data-retention-dashboard">
      <div className="dashboard-header">
        <div className="header-content">
          <h1><FaShieldAlt /> Data Retention & Right-to-Erasure</h1>
          <p>Manage data retention policies and user privacy rights</p>
        </div>
        {stats && (
          <div className="stats-summary">
            <div className="stat-item">
              <span className="stat-number">{stats.active_policies}</span>
              <span className="stat-label">Active Policies</span>
            </div>
            <div className="stat-item">
              <span className="stat-number">{stats.pending_erasure_requests}</span>
              <span className="stat-label">Pending Requests</span>
            </div>
            <div className="stat-item">
              <span className="stat-number">{stats.storage_saved_gb.toFixed(1)}GB</span>
              <span className="stat-label">Storage Saved</span>
            </div>
          </div>
        )}
      </div>

      <div className="dashboard-tabs">
        <button 
          className={`tab ${activeTab === 'overview' ? 'active' : ''}`}
          onClick={() => setActiveTab('overview')}
        >
          <FaChartBar /> Overview
        </button>
        <button 
          className={`tab ${activeTab === 'policies' ? 'active' : ''}`}
          onClick={() => setActiveTab('policies')}
        >
          <FaCog /> Retention Policies
        </button>
        <button 
          className={`tab ${activeTab === 'erasure' ? 'active' : ''}`}
          onClick={() => setActiveTab('erasure')}
        >
          <FaTrashAlt /> Erasure Requests
        </button>
        <button 
          className={`tab ${activeTab === 'exports' ? 'active' : ''}`}
          onClick={() => setActiveTab('exports')}
        >
          <FaDownload /> Data Exports
        </button>
        <button 
          className={`tab ${activeTab === 'user-data' ? 'active' : ''}`}
          onClick={() => setActiveTab('user-data')}
        >
          <FaUser /> My Data
        </button>
      </div>

      {activeTab === 'overview' && stats && (
        <div className="overview-section">
          <div className="stats-grid">
            <div className="stat-card">
              <div className="stat-icon policies">
                <FaCog />
              </div>
              <div className="stat-content">
                <h3>{stats.total_policies}</h3>
                <p>Total Policies</p>
                <span className="stat-detail">{stats.active_policies} active</span>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon erasure">
                <FaTrashAlt />
              </div>
              <div className="stat-content">
                <h3>{stats.pending_erasure_requests}</h3>
                <p>Pending Erasure</p>
                <span className="stat-detail">{stats.completed_erasures} completed</span>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon exports">
                <FaFileExport />
              </div>
              <div className="stat-content">
                <h3>{stats.data_exports_this_month}</h3>
                <p>Exports This Month</p>
                <span className="stat-detail">Data portability</span>
              </div>
            </div>

            <div className="stat-card">
              <div className="stat-icon storage">
                <FaDatabase />
              </div>
              <div className="stat-content">
                <h3>{stats.storage_saved_gb.toFixed(1)}GB</h3>
                <p>Storage Saved</p>
                <span className="stat-detail">Through cleanup</span>
              </div>
            </div>
          </div>

          <div className="overview-cards">
            <div className="overview-card">
              <div className="card-header">
                <h3><FaUserShield /> Privacy Compliance</h3>
              </div>
              <div className="card-content">
                <div className="compliance-item">
                  <span className="compliance-label">GDPR Right to Erasure</span>
                  <span className="compliance-status active">✓ Active</span>
                </div>
                <div className="compliance-item">
                  <span className="compliance-label">Data Retention Policies</span>
                  <span className="compliance-status active">✓ Configured</span>
                </div>
                <div className="compliance-item">
                  <span className="compliance-label">Automated Cleanup</span>
                  <span className="compliance-status active">✓ Running</span>
                </div>
                <div className="compliance-item">
                  <span className="compliance-label">Data Portability</span>
                  <span className="compliance-status active">✓ Available</span>
                </div>
              </div>
            </div>

            <div className="overview-card">
              <div className="card-header">
                <h3><FaCalendarAlt /> Recent Activity</h3>
              </div>
              <div className="card-content">
                <div className="activity-item">
                  <span className="activity-time">Today</span>
                  <span className="activity-desc">Automated data cleanup completed</span>
                </div>
                <div className="activity-item">
                  <span className="activity-time">Yesterday</span>
                  <span className="activity-desc">2 erasure requests processed</span>
                </div>
                <div className="activity-item">
                  <span className="activity-time">2 days ago</span>
                  <span className="activity-desc">New retention policy created</span>
                </div>
                <div className="activity-item">
                  <span className="activity-time">3 days ago</span>
                  <span className="activity-desc">5 data exports completed</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      )}

      {activeTab === 'policies' && <RetentionPolicies />}
      {activeTab === 'erasure' && <ErasureRequests />}
      {activeTab === 'exports' && <DataExports />}
      {activeTab === 'user-data' && <UserDataSummary />}
    </div>
  );
};

export default DataRetentionDashboard;