import React, { useState } from 'react';
import { 
  FaTrashAlt, FaSpinner, FaCheck, FaTimes, FaEye, FaPlus,
  FaClock, FaCheckCircle, FaTimesCircle, FaExclamationTriangle
} from 'react-icons/fa';
import { useErasureRequests } from '../../hooks/useDataRetention';
import { ErasureRequest } from '../../types/dataRetention';

const ErasureRequests: React.FC = () => {
  const [filters, setFilters] = useState({ status: '', page: 1, page_size: 20 });
  const { requests, pagination, loading, error, createRequest, updateRequest } = useErasureRequests(filters);
  const [showCreateForm, setShowCreateForm] = useState(false);
  const [selectedRequest, setSelectedRequest] = useState<ErasureRequest | null>(null);
  const [formData, setFormData] = useState({
    request_type: 'partial' as 'partial' | 'complete',
    reason: '',
    data_categories: [] as string[]
  });
  const [processingNotes, setProcessingNotes] = useState('');

  const availableCategories = [
    'property_views',
    'search_history',
    'saved_properties',
    'notifications',
    'chat_messages',
    'activity_logs'
  ];

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      await createRequest(formData);
      setShowCreateForm(false);
      setFormData({
        request_type: 'partial',
        reason: '',
        data_categories: []
      });
    } catch (err) {
      console.error('Failed to create erasure request:', err);
    }
  };

  const handleApprove = async (request: ErasureRequest) => {
    try {
      await updateRequest(request.id, { 
        status: 'approved',
        notes: processingNotes 
      });
      setSelectedRequest(null);
      setProcessingNotes('');
    } catch (err) {
      console.error('Failed to approve request:', err);
    }
  };

  const handleReject = async (request: ErasureRequest) => {
    try {
      await updateRequest(request.id, { 
        status: 'rejected',
        notes: processingNotes 
      });
      setSelectedRequest(null);
      setProcessingNotes('');
    } catch (err) {
      console.error('Failed to reject request:', err);
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending': return <FaClock className="status-icon pending" />;
      case 'approved': return <FaCheck className="status-icon approved" />;
      case 'rejected': return <FaTimes className="status-icon rejected" />;
      case 'completed': return <FaCheckCircle className="status-icon completed" />;
      default: return <FaExclamationTriangle className="status-icon unknown" />;
    }
  };

  const getStatusClass = (status: string) => {
    return `status-badge ${status}`;
  };

  if (loading) {
    return (
      <div className="erasure-requests">
        <div className="loading-spinner">
          <FaSpinner className="spinning" />
          <p>Loading erasure requests...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="erasure-requests">
        <div className="error-message">
          <p>Error loading requests: {error}</p>
        </div>
      </div>
    );
  }

  return (
    <div className="erasure-requests">
      <div className="requests-header">
        <h2>Erasure Requests</h2>
        <div className="header-actions">
          <select
            value={filters.status}
            onChange={(e) => setFilters({ ...filters, status: e.target.value, page: 1 })}
            className="status-filter"
          >
            <option value="">All Statuses</option>
            <option value="pending">Pending</option>
            <option value="approved">Approved</option>
            <option value="rejected">Rejected</option>
            <option value="completed">Completed</option>
          </select>
          <button 
            className="btn btn-primary"
            onClick={() => setShowCreateForm(true)}
          >
            <FaPlus /> New Request
          </button>
        </div>
      </div>

      {showCreateForm && (
        <div className="request-form-container">
          <form onSubmit={handleSubmit} className="request-form">
            <h3>Create Erasure Request</h3>
            
            <div className="form-group">
              <label>Request Type</label>
              <div className="radio-group">
                <label>
                  <input
                    type="radio"
                    value="partial"
                    checked={formData.request_type === 'partial'}
                    onChange={(e) => setFormData({ ...formData, request_type: e.target.value as 'partial' | 'complete' })}
                  />
                  Partial Erasure (specific data categories)
                </label>
                <label>
                  <input
                    type="radio"
                    value="complete"
                    checked={formData.request_type === 'complete'}
                    onChange={(e) => setFormData({ ...formData, request_type: e.target.value as 'partial' | 'complete' })}
                  />
                  Complete Erasure (all data)
                </label>
              </div>
            </div>

            {formData.request_type === 'partial' && (
              <div className="form-group">
                <label>Data Categories to Erase</label>
                <div className="checkbox-group">
                  {availableCategories.map((category) => (
                    <label key={category}>
                      <input
                        type="checkbox"
                        checked={formData.data_categories.includes(category)}
                        onChange={(e) => {
                          if (e.target.checked) {
                            setFormData({
                              ...formData,
                              data_categories: [...formData.data_categories, category]
                            });
                          } else {
                            setFormData({
                              ...formData,
                              data_categories: formData.data_categories.filter(c => c !== category)
                            });
                          }
                        }}
                      />
                      {category.replace('_', ' ').toUpperCase()}
                    </label>
                  ))}
                </div>
              </div>
            )}

            <div className="form-group">
              <label htmlFor="reason">Reason for Erasure</label>
              <textarea
                id="reason"
                value={formData.reason}
                onChange={(e) => setFormData({ ...formData, reason: e.target.value })}
                placeholder="Please explain why you want to erase this data..."
                rows={4}
                required
              />
            </div>

            <div className="form-actions">
              <button type="submit" className="btn btn-primary">
                <FaTrashAlt /> Submit Request
              </button>
              <button 
                type="button" 
                className="btn btn-secondary" 
                onClick={() => setShowCreateForm(false)}
              >
                Cancel
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="requests-list">
        {requests.length === 0 ? (
          <div className="empty-state">
            <p>No erasure requests found.</p>
          </div>
        ) : (
          <>
            <div className="requests-grid">
              {requests.map((request) => (
                <div key={request.id} className="request-card">
                  <div className="request-header">
                    <div className="request-id">Request #{request.id}</div>
                    <div className={getStatusClass(request.status)}>
                      {getStatusIcon(request.status)}
                      {request.status.toUpperCase()}
                    </div>
                  </div>
                  
                  <div className="request-content">
                    <div className="request-detail">
                      <span className="label">Type:</span>
                      <span className="value">{request.request_type.toUpperCase()}</span>
                    </div>
                    
                    <div className="request-detail">
                      <span className="label">Requested:</span>
                      <span className="value">{new Date(request.requested_at).toLocaleDateString()}</span>
                    </div>
                    
                    {request.data_categories.length > 0 && (
                      <div className="request-detail">
                        <span className="label">Categories:</span>
                        <div className="categories">
                          {request.data_categories.map((category) => (
                            <span key={category} className="category-tag">
                              {category.replace('_', ' ')}
                            </span>
                          ))}
                        </div>
                      </div>
                    )}
                    
                    <div className="request-reason">
                      <span className="label">Reason:</span>
                      <p>{request.reason}</p>
                    </div>
                    
                    {request.notes && (
                      <div className="request-notes">
                        <span className="label">Notes:</span>
                        <p>{request.notes}</p>
                      </div>
                    )}
                  </div>
                  
                  <div className="request-actions">
                    <button
                      className="btn btn-sm btn-secondary"
                      onClick={() => setSelectedRequest(request)}
                    >
                      <FaEye /> View Details
                    </button>
                    
                    {request.status === 'pending' && (
                      <>
                        <button
                          className="btn btn-sm btn-success"
                          onClick={() => {
                            setSelectedRequest(request);
                            setProcessingNotes('');
                          }}
                        >
                          <FaCheck /> Approve
                        </button>
                        <button
                          className="btn btn-sm btn-danger"
                          onClick={() => {
                            setSelectedRequest(request);
                            setProcessingNotes('');
                          }}
                        >
                          <FaTimes /> Reject
                        </button>
                      </>
                    )}
                  </div>
                </div>
              ))}
            </div>

            {/* Pagination */}
            {pagination.total_count > pagination.page_size && (
              <div className="pagination">
                <button
                  disabled={!pagination.has_previous}
                  onClick={() => setFilters({ ...filters, page: filters.page - 1 })}
                  className="btn btn-secondary"
                >
                  Previous
                </button>
                <span className="page-info">
                  Page {pagination.page} of {Math.ceil(pagination.total_count / pagination.page_size)}
                </span>
                <button
                  disabled={!pagination.has_next}
                  onClick={() => setFilters({ ...filters, page: filters.page + 1 })}
                  className="btn btn-secondary"
                >
                  Next
                </button>
              </div>
            )}
          </>
        )}
      </div>

      {/* Request Details Modal */}
      {selectedRequest && (
        <div className="modal-overlay" onClick={() => setSelectedRequest(null)}>
          <div className="modal-content" onClick={(e) => e.stopPropagation()}>
            <div className="modal-header">
              <h3>Erasure Request #{selectedRequest.id}</h3>
              <button 
                className="modal-close"
                onClick={() => setSelectedRequest(null)}
              >
                <FaTimes />
              </button>
            </div>
            
            <div className="modal-body">
              <div className="request-details">
                <div className="detail-row">
                  <span className="label">Status:</span>
                  <span className={getStatusClass(selectedRequest.status)}>
                    {getStatusIcon(selectedRequest.status)}
                    {selectedRequest.status.toUpperCase()}
                  </span>
                </div>
                
                <div className="detail-row">
                  <span className="label">Type:</span>
                  <span>{selectedRequest.request_type.toUpperCase()}</span>
                </div>
                
                <div className="detail-row">
                  <span className="label">Requested At:</span>
                  <span>{new Date(selectedRequest.requested_at).toLocaleString()}</span>
                </div>
                
                {selectedRequest.processed_at && (
                  <div className="detail-row">
                    <span className="label">Processed At:</span>
                    <span>{new Date(selectedRequest.processed_at).toLocaleString()}</span>
                  </div>
                )}
                
                {selectedRequest.completion_date && (
                  <div className="detail-row">
                    <span className="label">Completed At:</span>
                    <span>{new Date(selectedRequest.completion_date).toLocaleString()}</span>
                  </div>
                )}
                
                <div className="detail-row">
                  <span className="label">Reason:</span>
                  <p>{selectedRequest.reason}</p>
                </div>
                
                {selectedRequest.data_categories.length > 0 && (
                  <div className="detail-row">
                    <span className="label">Data Categories:</span>
                    <div className="categories">
                      {selectedRequest.data_categories.map((category) => (
                        <span key={category} className="category-tag">
                          {category.replace('_', ' ')}
                        </span>
                      ))}
                    </div>
                  </div>
                )}
                
                {selectedRequest.notes && (
                  <div className="detail-row">
                    <span className="label">Processing Notes:</span>
                    <p>{selectedRequest.notes}</p>
                  </div>
                )}
              </div>
              
              {selectedRequest.status === 'pending' && (
                <div className="processing-section">
                  <h4>Process Request</h4>
                  <textarea
                    value={processingNotes}
                    onChange={(e) => setProcessingNotes(e.target.value)}
                    placeholder="Add processing notes..."
                    rows={3}
                  />
                  <div className="processing-actions">
                    <button
                      className="btn btn-success"
                      onClick={() => handleApprove(selectedRequest)}
                    >
                      <FaCheck /> Approve Request
                    </button>
                    <button
                      className="btn btn-danger"
                      onClick={() => handleReject(selectedRequest)}
                    >
                      <FaTimes /> Reject Request
                    </button>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ErasureRequests;