import React, { useState } from 'react';
import { FaPlus, FaEdit, FaToggleOn, FaToggleOff, FaSpinner, FaSave, FaTimes } from 'react-icons/fa';
import { useRetentionPolicies } from '../../hooks/useDataRetention';
import { DataRetentionPolicy } from '../../types/dataRetention';

const RetentionPolicies: React.FC = () => {
  const { policies, loading, error, createPolicy, updatePolicy } = useRetentionPolicies();
  const [showCreateForm, setShowCreateForm] = useState(false);
  const [editingPolicy, setEditingPolicy] = useState<DataRetentionPolicy | null>(null);
  const [formData, setFormData] = useState({
    data_type: '',
    retention_period_days: 365,
    description: '',
    is_active: true
  });

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    try {
      if (editingPolicy) {
        await updatePolicy(editingPolicy.id, formData);
        setEditingPolicy(null);
      } else {
        await createPolicy(formData);
        setShowCreateForm(false);
      }
      setFormData({
        data_type: '',
        retention_period_days: 365,
        description: '',
        is_active: true
      });
    } catch (err) {
      console.error('Failed to save policy:', err);
    }
  };

  const handleEdit = (policy: DataRetentionPolicy) => {
    setEditingPolicy(policy);
    setFormData({
      data_type: policy.data_type,
      retention_period_days: policy.retention_period_days,
      description: policy.description,
      is_active: policy.is_active
    });
    setShowCreateForm(false);
  };

  const handleToggleActive = async (policy: DataRetentionPolicy) => {
    try {
      await updatePolicy(policy.id, { is_active: !policy.is_active });
    } catch (err) {
      console.error('Failed to toggle policy:', err);
    }
  };

  const cancelEdit = () => {
    setEditingPolicy(null);
    setShowCreateForm(false);
    setFormData({
      data_type: '',
      retention_period_days: 365,
      description: '',
      is_active: true
    });
  };

  if (loading) {
    return (
      <div className="retention-policies">
        <div className="loading-spinner">
          <FaSpinner className="spinning" />
          <p>Loading retention policies...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="retention-policies">
        <div className="error-message">
          <p>Error loading policies: {error}</p>
        </div>
      </div>
    );
  }

  return (
    <div className="retention-policies">
      <div className="policies-header">
        <h2>Data Retention Policies</h2>
        <button 
          className="btn btn-primary"
          onClick={() => setShowCreateForm(true)}
          disabled={showCreateForm || editingPolicy}
        >
          <FaPlus /> Create Policy
        </button>
      </div>

      {(showCreateForm || editingPolicy) && (
        <div className="policy-form-container">
          <form onSubmit={handleSubmit} className="policy-form">
            <h3>{editingPolicy ? 'Edit Policy' : 'Create New Policy'}</h3>
            
            <div className="form-group">
              <label htmlFor="data_type">Data Type</label>
              <input
                type="text"
                id="data_type"
                value={formData.data_type}
                onChange={(e) => setFormData({ ...formData, data_type: e.target.value })}
                placeholder="e.g., user_profiles, activity_logs"
                required
              />
            </div>

            <div className="form-group">
              <label htmlFor="retention_period_days">Retention Period (Days)</label>
              <input
                type="number"
                id="retention_period_days"
                value={formData.retention_period_days}
                onChange={(e) => setFormData({ ...formData, retention_period_days: parseInt(e.target.value) })}
                min="1"
                required
              />
            </div>

            <div className="form-group">
              <label htmlFor="description">Description</label>
              <textarea
                id="description"
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                placeholder="Describe what this policy covers..."
                rows={3}
                required
              />
            </div>

            <div className="form-group checkbox-group">
              <label>
                <input
                  type="checkbox"
                  checked={formData.is_active}
                  onChange={(e) => setFormData({ ...formData, is_active: e.target.checked })}
                />
                Active Policy
              </label>
            </div>

            <div className="form-actions">
              <button type="submit" className="btn btn-primary">
                <FaSave /> {editingPolicy ? 'Update' : 'Create'} Policy
              </button>
              <button type="button" className="btn btn-secondary" onClick={cancelEdit}>
                <FaTimes /> Cancel
              </button>
            </div>
          </form>
        </div>
      )}

      <div className="policies-list">
        {policies.length === 0 ? (
          <div className="empty-state">
            <p>No retention policies found. Create your first policy to get started.</p>
          </div>
        ) : (
          <div className="policies-grid">
            {policies.map((policy) => (
              <div key={policy.id} className={`policy-card ${!policy.is_active ? 'inactive' : ''}`}>
                <div className="policy-header">
                  <h3>{policy.data_type}</h3>
                  <div className="policy-actions">
                    <button
                      className="btn-icon"
                      onClick={() => handleEdit(policy)}
                      title="Edit policy"
                    >
                      <FaEdit />
                    </button>
                    <button
                      className={`btn-icon ${policy.is_active ? 'active' : 'inactive'}`}
                      onClick={() => handleToggleActive(policy)}
                      title={policy.is_active ? 'Deactivate policy' : 'Activate policy'}
                    >
                      {policy.is_active ? <FaToggleOn /> : <FaToggleOff />}
                    </button>
                  </div>
                </div>
                
                <div className="policy-content">
                  <div className="policy-detail">
                    <span className="label">Retention Period:</span>
                    <span className="value">{policy.retention_period_days} days</span>
                  </div>
                  
                  <div className="policy-detail">
                    <span className="label">Status:</span>
                    <span className={`status ${policy.is_active ? 'active' : 'inactive'}`}>
                      {policy.is_active ? 'Active' : 'Inactive'}
                    </span>
                  </div>
                  
                  <div className="policy-description">
                    <p>{policy.description}</p>
                  </div>
                  
                  <div className="policy-meta">
                    <small>Created: {new Date(policy.created_at).toLocaleDateString()}</small>
                    {policy.updated_at !== policy.created_at && (
                      <small>Updated: {new Date(policy.updated_at).toLocaleDateString()}</small>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
};

export default RetentionPolicies;