import React, { useState } from 'react';
import { 
  FaUser, FaDatabase, FaCalendarAlt, FaShieldAlt, FaTrashAlt, 
  FaDownload, FaSpinner, FaInfoCircle, FaExclamationTriangle
} from 'react-icons/fa';
import { useUserDataSummary, useDataExports, useErasureRequests } from '../../hooks/useDataRetention';

const UserDataSummary: React.FC = () => {
  const { summary, loading, error } = useUserDataSummary();
  const { requestExport } = useDataExports();
  const { createRequest } = useErasureRequests();
  const [showErasureForm, setShowErasureForm] = useState(false);
  const [erasureReason, setErasureReason] = useState('');
  const [selectedCategories, setSelectedCategories] = useState<string[]>([]);
  const [requesting, setRequesting] = useState(false);

  const handleQuickExport = async (exportType: 'personal_data' | 'complete') => {
    try {
      setRequesting(true);
      await requestExport(exportType);
      alert('Export request submitted successfully! You will be notified when it\'s ready.');
    } catch (err) {
      alert('Failed to request export. Please try again.');
    } finally {
      setRequesting(false);
    }
  };

  const handleErasureRequest = async (requestType: 'partial' | 'complete') => {
    if (requestType === 'partial' && selectedCategories.length === 0) {
      alert('Please select at least one data category to erase.');
      return;
    }

    if (!erasureReason.trim()) {
      alert('Please provide a reason for the erasure request.');
      return;
    }

    try {
      setRequesting(true);
      await createRequest({
        request_type: requestType,
        reason: erasureReason,
        data_categories: requestType === 'partial' ? selectedCategories : []
      });
      alert('Erasure request submitted successfully! It will be reviewed by our team.');
      setShowErasureForm(false);
      setErasureReason('');
      setSelectedCategories([]);
    } catch (err) {
      alert('Failed to submit erasure request. Please try again.');
    } finally {
      setRequesting(false);
    }
  };

  const formatDataSize = (sizeInMB: number) => {
    if (sizeInMB < 1) {
      return `${(sizeInMB * 1024).toFixed(0)} KB`;
    }
    return `${sizeInMB.toFixed(2)} MB`;
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'user_profiles': return <FaUser />;
      case 'property_views': return <FaDatabase />;
      case 'search_history': return <FaDatabase />;
      case 'visit_schedules': return <FaCalendarAlt />;
      default: return <FaDatabase />;
    }
  };

  if (loading) {
    return (
      <div className="user-data-summary">
        <div className="loading-spinner">
          <FaSpinner className="spinning" />
          <p>Loading your data summary...</p>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="user-data-summary">
        <div className="error-message">
          <p>Error loading data summary: {error}</p>
        </div>
      </div>
    );
  }

  if (!summary) {
    return (
      <div className="user-data-summary">
        <div className="error-message">
          <p>No data summary available.</p>
        </div>
      </div>
    );
  }

  return (
    <div className="user-data-summary">
      <div className="summary-header">
        <h2>My Data Summary</h2>
        <p>Overview of your personal data stored in our system</p>
      </div>

      <div className="user-info-card">
        <div className="user-info-header">
          <FaUser className="user-icon" />
          <div className="user-details">
            <h3>{summary.username || 'User'}</h3>
            {summary.email && <p className="user-email">{summary.email}</p>}
            {summary.phone_number && <p className="user-phone">{summary.phone_number}</p>}
          </div>
        </div>
        
        <div className="user-stats">
          <div className="stat-item">
            <span className="stat-label">Account Created</span>
            <span className="stat-value">
              <FaCalendarAlt />
              {new Date(summary.account_created).toLocaleDateString()}
            </span>
          </div>
          
          {summary.last_activity && (
            <div className="stat-item">
              <span className="stat-label">Last Activity</span>
              <span className="stat-value">
                <FaCalendarAlt />
                {new Date(summary.last_activity).toLocaleDateString()}
              </span>
            </div>
          )}
          
          <div className="stat-item">
            <span className="stat-label">Total Data Size</span>
            <span className="stat-value">
              <FaDatabase />
              {formatDataSize(summary.total_data_size_mb)}
            </span>
          </div>
        </div>
      </div>

      <div className="data-categories">
        <h3>Data Categories</h3>
        <div className="categories-grid">
          {summary.data_categories.map((category) => (
            <div key={category.category} className="category-card">
              <div className="category-header">
                <div className="category-icon">
                  {getCategoryIcon(category.category)}
                </div>
                <h4>{category.category.replace('_', ' ').toUpperCase()}</h4>
              </div>
              
              <div className="category-content">
                <p className="category-description">{category.description}</p>
                
                <div className="category-stats">
                  <div className="stat">
                    <span className="label">Records:</span>
                    <span className="value">{category.data_count.toLocaleString()}</span>
                  </div>
                  
                  <div className="stat">
                    <span className="label">Last Updated:</span>
                    <span className="value">
                      {new Date(category.last_updated).toLocaleDateString()}
                    </span>
                  </div>
                  
                  <div className="stat">
                    <span className="label">Retention:</span>
                    <span className="value">{category.retention_days} days</span>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>

      <div className="privacy-actions">
        <h3>Privacy Actions</h3>
        <div className="actions-grid">
          <div className="action-card">
            <div className="action-icon export">
              <FaDownload />
            </div>
            <h4>Export My Data</h4>
            <p>Download a copy of your personal data for portability and backup purposes.</p>
            <div className="action-buttons">
              <button
                className="btn btn-primary"
                onClick={() => handleQuickExport('personal_data')}
                disabled={requesting}
              >
                {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
                Export Personal Data
              </button>
              <button
                className="btn btn-secondary"
                onClick={() => handleQuickExport('complete')}
                disabled={requesting}
              >
                {requesting ? <FaSpinner className="spinning" /> : <FaDownload />}
                Export All Data
              </button>
            </div>
          </div>

          <div className="action-card">
            <div className="action-icon erasure">
              <FaTrashAlt />
            </div>
            <h4>Request Data Erasure</h4>
            <p>Request deletion of your personal data under your right to be forgotten.</p>
            <div className="action-buttons">
              <button
                className="btn btn-danger"
                onClick={() => setShowErasureForm(true)}
              >
                <FaTrashAlt />
                Request Erasure
              </button>
            </div>
          </div>

          <div className="action-card">
            <div className="action-icon privacy">
              <FaShieldAlt />
            </div>
            <h4>Privacy Settings</h4>
            <p>Manage your privacy preferences and consent settings.</p>
            <div className="action-buttons">
              <button className="btn btn-secondary">
                <FaShieldAlt />
                Manage Privacy
              </button>
            </div>
          </div>
        </div>
      </div>

      {summary.retention_eligible && (
        <div className="retention-notice">
          <FaInfoCircle />
          <div className="notice-content">
            <h4>Data Retention Notice</h4>
            <p>
              Some of your data is eligible for automatic cleanup based on our retention policies. 
              This helps keep your data current and reduces storage.
            </p>
          </div>
        </div>
      )}

      <div className="gdpr-info">
        <h3>Your Data Rights</h3>
        <div className="rights-grid">
          <div className="right-item">
            <FaInfoCircle />
            <div>
              <h4>Right to Access</h4>
              <p>You can request information about what personal data we hold about you.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaDownload />
            <div>
              <h4>Right to Portability</h4>
              <p>You can request a copy of your data in a machine-readable format.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaTrashAlt />
            <div>
              <h4>Right to Erasure</h4>
              <p>You can request deletion of your personal data under certain conditions.</p>
            </div>
          </div>
          
          <div className="right-item">
            <FaShieldAlt />
            <div>
              <h4>Right to Rectification</h4>
              <p>You can request correction of inaccurate or incomplete personal data.</p>
            </div>
          </div>
        </div>
      </div>

      {/* Erasure Request Modal */}
      {showErasureForm && (
        <div className="modal-overlay" onClick={() => setShowErasureForm(false)}>
          <div className="modal-content" onClick={(e) => e.stopPropagation()}>
            <div className="modal-header">
              <h3>Request Data Erasure</h3>
              <button 
                className="modal-close"
                onClick={() => setShowErasureForm(false)}
              >
                ×
              </button>
            </div>
            
            <div className="modal-body">
              <div className="warning-notice">
                <FaExclamationTriangle />
                <p>
                  <strong>Important:</strong> Data erasure is permanent and cannot be undone. 
                  Please consider exporting your data first if you might need it later.
                </p>
              </div>

              <div className="form-group">
                <label>Erasure Type</label>
                <div className="radio-group">
                  <label>
                    <input
                      type="radio"
                      name="erasureType"
                      value="partial"
                      onChange={() => {}}
                    />
                    Partial Erasure (select specific categories)
                  </label>
                  <label>
                    <input
                      type="radio"
                      name="erasureType"
                      value="complete"
                      onChange={() => {}}
                    />
                    Complete Erasure (all data)
                  </label>
                </div>
              </div>

              <div className="form-group">
                <label>Data Categories (for partial erasure)</label>
                <div className="checkbox-group">
                  {summary.data_categories.map((category) => (
                    <label key={category.category}>
                      <input
                        type="checkbox"
                        checked={selectedCategories.includes(category.category)}
                        onChange={(e) => {
                          if (e.target.checked) {
                            setSelectedCategories([...selectedCategories, category.category]);
                          } else {
                            setSelectedCategories(selectedCategories.filter(c => c !== category.category));
                          }
                        }}
                      />
                      {category.category.replace('_', ' ').toUpperCase()} ({category.data_count} records)
                    </label>
                  ))}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="erasureReason">Reason for Erasure *</label>
                <textarea
                  id="erasureReason"
                  value={erasureReason}
                  onChange={(e) => setErasureReason(e.target.value)}
                  placeholder="Please explain why you want to erase this data..."
                  rows={4}
                  required
                />
              </div>

              <div className="modal-actions">
                <button
                  className="btn btn-danger"
                  onClick={() => handleErasureRequest(selectedCategories.length > 0 ? 'partial' : 'complete')}
                  disabled={requesting}
                >
                  {requesting ? <FaSpinner className="spinning" /> : <FaTrashAlt />}
                  Submit Erasure Request
                </button>
                <button
                  className="btn btn-secondary"
                  onClick={() => setShowErasureForm(false)}
                >
                  Cancel
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default UserDataSummary;