import React, { useState } from 'react';
import DataRetentionAPI from '../../api/dataRetention';

const ExportDebugger: React.FC = () => {
  const [logs, setLogs] = useState<string[]>([]);
  const [loading, setLoading] = useState(false);

  const addLog = (message: string) => {
    const timestamp = new Date().toLocaleTimeString();
    setLogs(prev => [...prev, `[${timestamp}] ${message}`]);
  };

  const testExportFlow = async () => {
    setLoading(true);
    setLogs([]);
    
    try {
      addLog('🚀 Starting export test...');
      
      // Step 1: Request export
      addLog('📤 Requesting personal data export...');
      const exportResult = await DataRetentionAPI.requestDataExport('personal_data');
      
      addLog(`✅ Export created: ID ${exportResult.id}`);
      addLog(`📊 Status: ${exportResult.status}`);
      addLog(`📁 File path: ${exportResult.file_path || 'Not set'}`);
      addLog(`📏 File size: ${exportResult.file_size || 'Unknown'} bytes`);
      
      // Step 2: Check status and try download
      if (exportResult.status === 'completed') {
        addLog('⚡ Export completed immediately!');
        
        try {
          addLog('⬇️ Attempting download...');
          const blob = await DataRetentionAPI.downloadDataExport(exportResult.id);
          
          addLog(`✅ Download successful! Size: ${blob.size} bytes`);
          addLog(`📄 Type: ${blob.type}`);
          
          // Trigger actual download
          const url = window.URL.createObjectURL(blob);
          const a = document.createElement('a');
          a.href = url;
          a.download = `debug_export_${exportResult.id}.zip`;
          document.body.appendChild(a);
          a.click();
          window.URL.revokeObjectURL(url);
          document.body.removeChild(a);
          
          addLog('💾 File download triggered!');
          
        } catch (downloadError) {
          addLog(`❌ Download failed: ${downloadError}`);
        }
        
      } else {
        addLog(`⏳ Export not completed yet. Status: ${exportResult.status}`);
      }
      
    } catch (error) {
      addLog(`❌ Export request failed: ${error}`);
    } finally {
      setLoading(false);
    }
  };

  const clearLogs = () => {
    setLogs([]);
  };

  return (
    <div style={{ padding: '20px', maxWidth: '800px', margin: '0 auto' }}>
      <h2>🔧 Export Flow Debugger</h2>
      
      <div style={{ marginBottom: '20px' }}>
        <button 
          onClick={testExportFlow} 
          disabled={loading}
          style={{
            padding: '10px 20px',
            backgroundColor: loading ? '#ccc' : '#007bff',
            color: 'white',
            border: 'none',
            borderRadius: '5px',
            cursor: loading ? 'not-allowed' : 'pointer',
            marginRight: '10px'
          }}
        >
          {loading ? '🔄 Testing...' : '🧪 Test Export Flow'}
        </button>
        
        <button 
          onClick={clearLogs}
          style={{
            padding: '10px 20px',
            backgroundColor: '#6c757d',
            color: 'white',
            border: 'none',
            borderRadius: '5px',
            cursor: 'pointer'
          }}
        >
          🗑️ Clear Logs
        </button>
      </div>
      
      <div 
        style={{
          backgroundColor: '#f8f9fa',
          border: '1px solid #dee2e6',
          borderRadius: '5px',
          padding: '15px',
          minHeight: '300px',
          fontFamily: 'monospace',
          fontSize: '14px',
          whiteSpace: 'pre-wrap',
          overflow: 'auto'
        }}
      >
        {logs.length === 0 ? (
          <div style={{ color: '#6c757d', fontStyle: 'italic' }}>
            Click "Test Export Flow" to start debugging...
          </div>
        ) : (
          logs.map((log, index) => (
            <div key={index} style={{ marginBottom: '5px' }}>
              {log}
            </div>
          ))
        )}
      </div>
      
      <div style={{ marginTop: '20px', fontSize: '14px', color: '#6c757d' }}>
        <strong>What this test does:</strong>
        <ul>
          <li>Requests a personal data export</li>
          <li>Checks the export status</li>
          <li>If completed, attempts to download the file</li>
          <li>Logs each step for debugging</li>
        </ul>
      </div>
    </div>
  );
};

export default ExportDebugger;