import React, { useState } from 'react';
import { DocumentManager } from './DocumentManager';
import { uploadPropertyDocuments, getPropertyDocuments } from '../../api/documents';
import { FileText, Upload, Eye, Download } from 'lucide-react';
import { toast } from 'sonner';

/**
 * Example component showing how to integrate the document management system
 * with your existing property listing workflow
 */
export const DocumentIntegrationExample: React.FC = () => {
  const [selectedPropertyId, setSelectedPropertyId] = useState<string>('');
  const [showDocumentManager, setShowDocumentManager] = useState(false);

  // Example property data - replace with your actual property data
  const exampleProperties = [
    { id: 'prop_001', title: 'Luxury 3BHK Apartment', location: 'Bandra West, Mumbai' },
    { id: 'prop_002', title: 'Modern Villa', location: 'Whitefield, Bangalore' },
    { id: 'prop_003', title: 'Commercial Office Space', location: 'Cyber City, Gurgaon' }
  ];

  const handlePropertySelect = (propertyId: string) => {
    setSelectedPropertyId(propertyId);
    setShowDocumentManager(true);
  };

  // Example: Quick upload function
  const handleQuickUpload = async (propertyId: string, files: FileList) => {
    const fileArray = Array.from(files);
    const documentTypes = fileArray.map(() => 'other'); // Default type
    
    try {
      const result = await uploadPropertyDocuments(propertyId, fileArray, documentTypes);
      if (result.success) {
        toast.success(`${result.documents?.length} documents uploaded successfully`);
      } else {
        toast.error(result.error || 'Upload failed');
      }
    } catch (error) {
      toast.error('Upload failed');
    }
  };

  // Example: Fetch documents for a property
  const handleFetchDocuments = async (propertyId: string) => {
    try {
      const documents = await getPropertyDocuments(propertyId);
      console.log('Property documents:', documents);
      toast.success(`Found ${documents.length} documents`);
    } catch (error) {
      toast.error('Failed to fetch documents');
    }
  };

  return (
    <div className="max-w-4xl mx-auto p-6 space-y-8">
      <div className="text-center">
        <h1 className="text-3xl font-bold text-gray-900 mb-4">
          Document Management Integration Example
        </h1>
        <p className="text-gray-600 max-w-2xl mx-auto">
          This example shows how to integrate the document management system with your property listings.
          Select a property to manage its documents.
        </p>
      </div>

      {/* Property Selection */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <h2 className="text-xl font-semibold text-gray-900 mb-4">Select Property</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          {exampleProperties.map((property) => (
            <div
              key={property.id}
              className="border border-gray-200 rounded-lg p-4 hover:border-blue-300 hover:shadow-md transition-all cursor-pointer"
              onClick={() => handlePropertySelect(property.id)}
            >
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <h3 className="font-medium text-gray-900 mb-1">{property.title}</h3>
                  <p className="text-sm text-gray-600">{property.location}</p>
                  <p className="text-xs text-gray-500 mt-2">ID: {property.id}</p>
                </div>
                <FileText className="w-5 h-5 text-gray-400" />
              </div>
              
              <div className="mt-4 flex space-x-2">
                <button
                  onClick={(e) => {
                    e.stopPropagation();
                    handleFetchDocuments(property.id);
                  }}
                  className="flex-1 px-3 py-2 text-xs bg-blue-50 text-blue-700 rounded-md hover:bg-blue-100 transition-colors flex items-center justify-center"
                >
                  <Eye className="w-3 h-3 mr-1" />
                  View Docs
                </button>
                
                <label className="flex-1 px-3 py-2 text-xs bg-green-50 text-green-700 rounded-md hover:bg-green-100 transition-colors cursor-pointer flex items-center justify-center">
                  <Upload className="w-3 h-3 mr-1" />
                  Quick Upload
                  <input
                    type="file"
                    multiple
                    className="hidden"
                    onChange={(e) => {
                      if (e.target.files) {
                        handleQuickUpload(property.id, e.target.files);
                      }
                    }}
                  />
                </label>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* API Usage Examples */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h2 className="text-xl font-semibold text-gray-900 mb-4">API Usage Examples</h2>
        <div className="space-y-4 text-sm">
          <div className="bg-white rounded-md p-4 border-l-4 border-blue-500">
            <h3 className="font-medium text-gray-900 mb-2">Upload Documents</h3>
            <pre className="text-gray-600 text-xs overflow-x-auto">
{`import { uploadPropertyDocuments } from './api/documents';

const files = [file1, file2]; // File objects
const types = ['ownership', 'noc']; // Document types
const result = await uploadPropertyDocuments(propertyId, files, types);`}
            </pre>
          </div>
          
          <div className="bg-white rounded-md p-4 border-l-4 border-green-500">
            <h3 className="font-medium text-gray-900 mb-2">Get Property Documents</h3>
            <pre className="text-gray-600 text-xs overflow-x-auto">
{`import { getPropertyDocuments } from './api/documents';

const documents = await getPropertyDocuments(propertyId);
console.log(documents); // Array of PropertyDocument objects`}
            </pre>
          </div>
          
          <div className="bg-white rounded-md p-4 border-l-4 border-yellow-500">
            <h3 className="font-medium text-gray-900 mb-2">Generate Download URL</h3>
            <pre className="text-gray-600 text-xs overflow-x-auto">
{`import { generateDocumentDownloadURL } from './api/documents';

const result = await generateDocumentDownloadURL(documentId);
if (result.success) {
  window.open(result.download_url, '_blank');
}`}
            </pre>
          </div>
          
          <div className="bg-white rounded-md p-4 border-l-4 border-red-500">
            <h3 className="font-medium text-gray-900 mb-2">Replace Document</h3>
            <pre className="text-gray-600 text-xs overflow-x-auto">
{`import { replaceDocument } from './api/documents';

const newFile = file; // New File object
const result = await replaceDocument(documentId, newFile, 'ownership');`}
            </pre>
          </div>
        </div>
      </div>

      {/* Document Manager Modal */}
      {showDocumentManager && selectedPropertyId && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            <div className="sticky top-0 bg-white border-b border-gray-200 px-6 py-4 flex items-center justify-between">
              <h2 className="text-xl font-semibold text-gray-900">Document Manager</h2>
              <button
                onClick={() => setShowDocumentManager(false)}
                className="text-gray-400 hover:text-gray-600 transition-colors"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>
            
            <div className="p-6">
              <DocumentManager
                propertyId={selectedPropertyId}
                propertyTitle={exampleProperties.find(p => p.id === selectedPropertyId)?.title}
                readOnly={false}
              />
            </div>
          </div>
        </div>
      )}
    </div>
  );
};