import React, { useState, useEffect } from 'react';
import { DocumentUpload } from './DocumentUpload';
import { getPropertyDocuments, generateDocumentDownloadURL, PropertyDocument } from '../../api/documents';
import { FileText, Download, Eye, Calendar, User } from 'lucide-react';
import { formatDate } from '@/utils/regionalFormatters';
import { toast } from 'sonner';

interface DocumentManagerProps {
  propertyId: string;
  propertyTitle?: string;
  readOnly?: boolean;
}

export const DocumentManager: React.FC<DocumentManagerProps> = ({
  propertyId,
  propertyTitle,
  readOnly = false
}) => {
  const [documents, setDocuments] = useState<PropertyDocument[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [selectedDocument, setSelectedDocument] = useState<PropertyDocument | null>(null);

  useEffect(() => {
    loadDocuments();
  }, [propertyId]);

  const loadDocuments = async () => {
    try {
      setIsLoading(true);
      const docs = await getPropertyDocuments(propertyId);
      setDocuments(docs);
    } catch (error) {
      console.error('Error loading documents:', error);
      toast.error('Failed to load documents');
    } finally {
      setIsLoading(false);
    }
  };

  const handleDocumentsChange = (newDocuments: PropertyDocument[]) => {
    setDocuments(newDocuments);
  };

  const handleDownload = async (document: PropertyDocument) => {
    try {
      const result = await generateDocumentDownloadURL(document.id);
      if (result.success && result.download_url) {
        // Create a temporary link to download the file
        const link = document.createElement('a');
        link.href = result.download_url;
        link.download = document.file_name;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        toast.success('Download started');
      } else {
        toast.error('Failed to generate download link');
      }
    } catch (error) {
      console.error('Download error:', error);
      toast.error('Download failed');
    }
  };

  const getDocumentIcon = (documentType: string) => {
    switch (documentType) {
      case 'ownership':
        return '🏠';
      case 'building_approval':
        return '🏗️';
      case 'noc':
        return '✅';
      case 'property_tax':
        return '💰';
      case 'encumbrance':
        return '📋';
      case 'lease_agreement':
        return '📄';
      default:
        return '📁';
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  if (isLoading) {
    return (
      <div className="flex items-center justify-center py-12">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
        <span className="ml-3 text-gray-600">Loading documents...</span>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <div>
          <h2 className="text-2xl font-bold text-gray-900">Document Manager</h2>
          {propertyTitle && (
            <p className="text-gray-600 mt-1">Managing documents for: {propertyTitle}</p>
          )}
        </div>
        <div className="text-sm text-gray-500">
          {documents.length} document{documents.length !== 1 ? 's' : ''} uploaded
        </div>
      </div>

      {/* Upload Section - Only show if not read-only */}
      {!readOnly && (
        <DocumentUpload
          propertyId={propertyId}
          existingDocuments={documents}
          onDocumentsChange={handleDocumentsChange}
        />
      )}

      {/* Documents List */}
      {documents.length > 0 && (
        <div className="bg-white rounded-lg shadow-sm border border-gray-200">
          <div className="px-6 py-4 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900">Uploaded Documents</h3>
          </div>
          
          <div className="divide-y divide-gray-200">
            {documents.map((doc) => (
              <div key={doc.id} className="px-6 py-4 hover:bg-gray-50 transition-colors">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4 flex-1">
                    <div className="text-2xl">{getDocumentIcon(doc.document_type)}</div>
                    
                    <div className="flex-1 min-w-0">
                      <h4 className="text-sm font-medium text-gray-900 truncate">
                        {doc.file_name}
                      </h4>
                      <div className="flex items-center space-x-4 mt-1 text-xs text-gray-500">
                        <span className="flex items-center">
                          <FileText className="w-3 h-3 mr-1" />
                          {formatFileSize(doc.file_size)}
                        </span>
                        <span className="flex items-center">
                          <Calendar className="w-3 h-3 mr-1" />
                          {formatDate(doc.upload_date)}
                        </span>
                      </div>
                      <div className="mt-1">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                          {doc.document_type.replace('_', ' ').toUpperCase()}
                        </span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex items-center space-x-2">
                    <button
                      onClick={() => setSelectedDocument(doc)}
                      className="p-2 text-gray-400 hover:text-blue-600 transition-colors"
                      title="View Details"
                    >
                      <Eye className="w-4 h-4" />
                    </button>
                    
                    <button
                      onClick={() => handleDownload(doc)}
                      className="p-2 text-gray-400 hover:text-green-600 transition-colors"
                      title="Download"
                    >
                      <Download className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Empty State */}
      {documents.length === 0 && !readOnly && (
        <div className="text-center py-12 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300">
          <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No documents uploaded yet</h3>
          <p className="text-gray-600">
            Upload property documents to help verify your listing and build trust with buyers.
          </p>
        </div>
      )}

      {/* Document Details Modal */}
      {selectedDocument && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg max-w-md w-full mx-4">
            <div className="px-6 py-4 border-b border-gray-200">
              <h3 className="text-lg font-semibold text-gray-900">Document Details</h3>
            </div>
            
            <div className="px-6 py-4 space-y-4">
              <div>
                <label className="text-sm font-medium text-gray-700">File Name</label>
                <p className="text-sm text-gray-900 mt-1">{selectedDocument.file_name}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-700">Document Type</label>
                <p className="text-sm text-gray-900 mt-1 capitalize">
                  {selectedDocument.document_type.replace('_', ' ')}
                </p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-700">File Size</label>
                <p className="text-sm text-gray-900 mt-1">{formatFileSize(selectedDocument.file_size)}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-700">Upload Date</label>
                <p className="text-sm text-gray-900 mt-1">
                  {new Date(selectedDocument.upload_date).toLocaleString()}
                </p>
              </div>
            </div>
            
            <div className="px-6 py-4 border-t border-gray-200 flex justify-end space-x-3">
              <button
                onClick={() => setSelectedDocument(null)}
                className="px-4 py-2 text-gray-700 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Close
              </button>
              <button
                onClick={() => {
                  handleDownload(selectedDocument);
                  setSelectedDocument(null);
                }}
                className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors flex items-center"
              >
                <Download className="w-4 h-4 mr-2" />
                Download
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};