import React, { useState, useRef, useCallback } from 'react';
import { Upload, FileText, X, Download, Eye, AlertCircle, CheckCircle, Loader2 } from 'lucide-react';
import { uploadPropertyDocuments, PropertyDocument, replaceDocument, deleteDocument } from '../../api/documents';
import { toast } from 'sonner';
import DateDisplay from '../regional/DateDisplay';

interface DocumentUploadProps {
  propertyId?: string;
  existingDocuments?: PropertyDocument[];
  onDocumentsChange?: (documents: PropertyDocument[]) => void;
  maxFileSize?: number; // in MB
  allowedTypes?: string[];
}

interface DocumentFile {
  file: File;
  type: string;
  preview?: string;
  uploading?: boolean;
  uploaded?: boolean;
  error?: string;
  id?: string;
  isEncrypted?: boolean;
  password?: string;
}

const DOCUMENT_TYPES = [
  { value: 'ownership', label: 'Property Ownership Documents', description: 'Sale deed, title deed, ownership certificate' },
  { value: 'building_approval', label: 'Building Approvals', description: 'Construction permit, building plan approval' },
  { value: 'noc', label: 'NOC Certificate', description: 'No Objection Certificate from authorities' },
  { value: 'property_tax', label: 'Property Tax Receipt', description: 'Latest property tax payment receipt' },
  { value: 'encumbrance', label: 'Encumbrance Certificate', description: 'Property transaction history certificate' },
  { value: 'lease_agreement', label: 'Lease Agreement', description: 'Lease agreement document' },
  { value: 'other', label: 'Other Documents', description: 'Any other relevant property documents' }
];

export const DocumentUpload: React.FC<DocumentUploadProps> = ({
  propertyId,
  existingDocuments = [],
  onDocumentsChange,
  maxFileSize = 10,
  allowedTypes = ['.pdf', '.doc', '.docx', '.jpg', '.jpeg', '.png']
}) => {
  const [documents, setDocuments] = useState<DocumentFile[]>([]);
  const [uploadedDocuments, setUploadedDocuments] = useState<PropertyDocument[]>(existingDocuments);
  const [isUploading, setIsUploading] = useState(false);
  const fileInputRef = useRef<HTMLInputElement>(null);

  const handleFileSelect = useCallback((files: FileList | null) => {
    if (!files) return;

    const newDocuments: DocumentFile[] = [];

    Array.from(files).forEach((file) => {
      // Validate file size
      if (file.size > maxFileSize * 1024 * 1024) {
        toast.error(`File ${file.name} is too large. Maximum size is ${maxFileSize}MB`);
        return;
      }

      // Validate file type
      const fileExtension = '.' + file.name.split('.').pop()?.toLowerCase();
      if (!allowedTypes.includes(fileExtension)) {
        toast.error(`File type ${fileExtension} is not allowed`);
        return;
      }

      newDocuments.push({
        file,
        type: 'other', // Default type
        preview: file.type.startsWith('image/') ? URL.createObjectURL(file) : undefined
      });
    });

    setDocuments(prev => [...prev, ...newDocuments]);
  }, [maxFileSize, allowedTypes]);

  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    handleFileSelect(e.dataTransfer.files);
  }, [handleFileSelect]);

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault();
  }, []);

  const removeDocument = (index: number) => {
    setDocuments(prev => {
      const newDocs = [...prev];
      if (newDocs[index].preview) {
        URL.revokeObjectURL(newDocs[index].preview!);
      }
      newDocs.splice(index, 1);
      return newDocs;
    });
  };

  const updateDocumentType = (index: number, type: string) => {
    setDocuments(prev => {
      const newDocs = [...prev];
      newDocs[index] = { ...newDocs[index], type };
      return newDocs;
    });
  };

  const updateDocumentEncryption = (index: number, isEncrypted: boolean, password?: string) => {
    setDocuments(prev => {
      const newDocs = [...prev];
      newDocs[index] = { ...newDocs[index], isEncrypted, password: isEncrypted ? password : undefined };
      return newDocs;
    });
  };

  const uploadDocuments = async () => {
    if (!propertyId) {
      toast.error('Property ID is required to upload documents');
      return;
    }

    if (documents.length === 0) {
      toast.error('Please select documents to upload');
      return;
    }

    // Validate encrypted documents have passwords
    const invalidDocs = documents.filter(doc => doc.isEncrypted && (!doc.password || doc.password.length < 6));
    if (invalidDocs.length > 0) {
      toast.error('Please set valid passwords (min 6 characters) for encrypted documents');
      return;
    }

    setIsUploading(true);

    try {
      // Upload documents one by one to handle passwords
      const uploadPromises = documents.map(async (doc) => {
        const formData = new FormData();
        formData.append('file', doc.file);
        formData.append('property_id', propertyId);
        formData.append('user_id', 'current_user_id'); // Replace with actual user ID
        formData.append('doc_type', doc.type);
        
        if (doc.isEncrypted && doc.password) {
          formData.append('password', doc.password);
        }

        const response = await fetch('/api/documents/upload', {
          method: 'POST',
          body: formData,
          headers: {
            'Authorization': `Bearer ${localStorage.getItem('access_token')}`
          }
        });

        if (!response.ok) {
          throw new Error(`Upload failed for ${doc.file.name}`);
        }

        const result = await response.json();
        
        // Map API response to frontend expected structure
        return {
          id: result.id,
          file_name: doc.file.name, // Use original filename
          document_type: result.doc_type || doc.type,
          file_size: doc.file.size, // Use original file size
          upload_date: result.created_at || new Date().toISOString(),
          is_encrypted: doc.isEncrypted || false,
          download_url: result.file_url,
          property_id: result.property_id
        };
      });

      const results = await Promise.all(uploadPromises);
      
      // Update uploaded documents with new results
      const newUploadedDocs = [...uploadedDocuments, ...results];
      setUploadedDocuments(newUploadedDocs);
      setDocuments([]);
      onDocumentsChange?.(newUploadedDocs);
      
      // Ask for password protection for each uploaded document
      for (const result of results) {
        const shouldProtect = confirm(`📄 Document "${result.file_name}" uploaded successfully!\n\n🔒 Do you want to set a password to protect this document?`);
        
        if (shouldProtect) {
          const password = prompt(`🔑 Set password for "${result.file_name}":\n\nPassword must be at least 6 characters.`);
          
          if (password && password.length >= 6) {
            // TODO: Call API to encrypt existing document
            toast.success(`🔒 Password protection set for ${result.file_name}`);
          } else if (password) {
            toast.error('❌ Password must be at least 6 characters');
          }
        }
      }
      
      const encryptedCount = documents.filter(doc => doc.isEncrypted).length;
      const message = encryptedCount > 0 
        ? `${results.length} document(s) uploaded successfully (${encryptedCount} encrypted 🔒)`
        : `${results.length} document(s) uploaded successfully`;
      
      toast.success(message);
    } catch (error) {
      console.error('Upload error:', error);
      toast.error(`Upload failed: ${error.message || 'Please try again.'}`);
      
      // Log more details for debugging
      console.log('Documents being uploaded:', documents);
      console.log('Property ID:', propertyId);
    } finally {
      setIsUploading(false);
    }
  };

  const handleReplaceDocument = async (documentId: string, newFile: File, documentType: string) => {
    try {
      const result = await replaceDocument(documentId, newFile, documentType);

      if (result.success) {
        // Update the document in the list
        setUploadedDocuments(prev =>
          prev.map(doc =>
            doc.id === documentId
              ? { ...doc, file_name: newFile.name, upload_date: new Date().toISOString() }
              : doc
          )
        );
        toast.success('Document replaced successfully');
      } else {
        toast.error(result.error || 'Replace failed');
      }
    } catch (error) {
      console.error('Replace error:', error);
      toast.error('Replace failed. Please try again.');
    }
  };

  const handleDeleteDocument = async (documentId: string) => {
    if (!confirm('Are you sure you want to delete this document?')) return;

    try {
      const result = await deleteDocument(documentId);

      if (result.success) {
        setUploadedDocuments(prev => prev.filter(doc => doc.id !== documentId));
        onDocumentsChange?.(uploadedDocuments.filter(doc => doc.id !== documentId));
        toast.success('Document deleted successfully');
      } else {
        toast.error(result.error || 'Delete failed');
      }
    } catch (error) {
      console.error('Delete error:', error);
      toast.error('Delete failed. Please try again.');
    }
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const getDocumentTypeLabel = (type: string) => {
    return DOCUMENT_TYPES.find(dt => dt.value === type)?.label || 'Other Documents';
  };

  return (
    <div className="space-y-6">
      {/* Upload Area */}
      <div className="bg-white rounded-lg border-2 border-dashed border-gray-300 hover:border-blue-400 transition-colors">
        <div
          className="p-8 text-center cursor-pointer"
          onDrop={handleDrop}
          onDragOver={handleDragOver}
          onClick={() => fileInputRef.current?.click()}
        >
          <Upload className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">Upload Property Documents</h3>
          <p className="text-gray-600 mb-4">
            Drag and drop files here, or click to select files
          </p>
          <p className="text-sm text-gray-500">
            Supported formats: {allowedTypes.join(', ')} • Max size: {maxFileSize}MB per file
          </p>

          <input
            ref={fileInputRef}
            type="file"
            multiple
            accept={allowedTypes.join(',')}
            onChange={(e) => handleFileSelect(e.target.files)}
            className="hidden"
          />
        </div>
      </div>

      {/* Selected Documents for Upload */}
      {documents.length > 0 && (
        <div className="bg-blue-50 rounded-lg p-5">
          <div className="flex items-center justify-between mb-4">
            <h4 className="text-lg font-semibold text-gray-900">Selected Documents ({documents.length})</h4>
            <button
              onClick={uploadDocuments}
              disabled={isUploading}
              className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed flex items-center"
            >
              {isUploading ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Uploading...
                </>
              ) : (
                <>
                  <Upload className="w-4 h-4 mr-2" />
                  Upload All
                </>
              )}
            </button>
          </div>

          <div className="space-y-3">
            {documents.map((doc, index) => (
              <div key={index} className="bg-white rounded-lg p-4 border border-gray-200">
                <div className="flex items-start justify-between">
                  <div className="flex items-start space-x-3 flex-1">
                    <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                      <FileText className="w-5 h-5 text-blue-600" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center space-x-2 mb-1">
                        <p className="font-medium text-gray-900 truncate">{doc.file.name}</p>
                        {doc.isEncrypted && (
                          <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                            🔒 Protected
                          </span>
                        )}
                      </div>
                      <p className="text-sm text-gray-500 mb-3">{formatFileSize(doc.file.size)}</p>

                      {/* Password Protection - More Prominent */}
                      <div className="mb-3 p-3 bg-blue-50 border-2 border-blue-200 rounded-lg">
                        <div className="flex items-center space-x-2 mb-2">
                          <input
                            type="checkbox"
                            id={`encrypt-${index}`}
                            checked={doc.isEncrypted || false}
                            onChange={(e) => {
                              if (e.target.checked) {
                                updateDocumentEncryption(index, true, '');
                              } else {
                                updateDocumentEncryption(index, false);
                              }
                            }}
                            className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                          />
                          <label htmlFor={`encrypt-${index}`} className="text-sm font-semibold text-blue-800">
                            🔒 Password Protect This Document
                          </label>
                        </div>
                        
                        {doc.isEncrypted && (
                          <div className="mt-2">
                            <label className="block text-sm font-semibold text-blue-800 mb-1">
                              🔑 Set Password for {doc.file.name}
                            </label>
                            <input
                              type="password"
                              value={doc.password || ''}
                              onChange={(e) => updateDocumentEncryption(index, true, e.target.value)}
                              placeholder="Enter password (min 6 characters)"
                              className="w-full px-3 py-2 border-2 border-blue-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 bg-white"
                              minLength={6}
                            />
                            {doc.password && doc.password.length < 6 && (
                              <p className="text-xs text-red-600 mt-1 font-medium">⚠️ Password must be at least 6 characters</p>
                            )}
                            {doc.password && doc.password.length >= 6 && (
                              <p className="text-xs text-green-600 mt-1 font-medium">✅ Password set successfully</p>
                            )}
                          </div>
                        )}
                      </div>

                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">
                          Document Type
                        </label>
                        <select
                          value={doc.type}
                          onChange={(e) => updateDocumentType(index, e.target.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        >
                          {DOCUMENT_TYPES.map(type => (
                            <option key={type.value} value={type.value}>
                              {type.label}
                            </option>
                          ))}
                        </select>
                      </div>
                    </div>
                  </div>

                  <button
                    onClick={() => removeDocument(index)}
                    className="ml-4 p-1 text-gray-400 hover:text-red-500 transition-colors"
                  >
                    <X className="w-5 h-5" />
                  </button>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Uploaded Documents */}
      {uploadedDocuments.length > 0 && (
        <div className="bg-green-50 rounded-lg p-5">
          <h4 className="text-lg font-semibold text-gray-900 mb-4">
            Uploaded Documents ({uploadedDocuments.length})
          </h4>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {uploadedDocuments.map((doc) => (
              <div key={doc.id} className="bg-white rounded-lg p-4 border border-gray-200">
                <div className="flex items-start justify-between">
                  <div className="flex items-start space-x-3 flex-1">
                    <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                      <CheckCircle className="w-5 h-5 text-green-600" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center space-x-2 mb-1">
                        <p className="font-medium text-gray-900 truncate">{doc.file_name}</p>
                        {doc.is_encrypted && (
                          <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                            🔒 Protected
                          </span>
                        )}
                      </div>
                      <p className="text-sm text-gray-500">{getDocumentTypeLabel(doc.document_type)}</p>
                      <p className="text-xs text-gray-400">
                        {formatFileSize(doc.file_size)} • <DateDisplay date={doc.upload_date} />
                      </p>
                      
                      {/* Password Protection Option - Always Show */}
                      <div className="mt-2 p-3 bg-blue-50 border-2 border-blue-200 rounded-lg">
                        {doc.is_encrypted ? (
                          <div className="text-xs text-green-600 font-medium flex items-center">
                            ✅ Password Protected
                          </div>
                        ) : (
                          <button
                            onClick={() => {
                              const password = prompt(`🔑 Set password for "${doc.file_name}":\n\nPassword must be at least 6 characters.`);
                              if (password && password.length >= 6) {
                                // TODO: Call API to encrypt existing document
                                toast.success(`🔒 Password protection added to ${doc.file_name}`);
                              } else if (password) {
                                toast.error('❌ Password must be at least 6 characters');
                              }
                            }}
                            className="w-full text-sm text-blue-600 hover:text-blue-800 font-semibold flex items-center justify-center py-2 px-3 bg-white border border-blue-300 rounded hover:bg-blue-50 transition-colors"
                          >
                            🔒 Set Password for this Document
                          </button>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="flex items-center space-x-2 ml-4">
                    <button
                      onClick={() => {
                        // Handle view/download
                        if (doc.download_url) {
                          window.open(doc.download_url, '_blank');
                        }
                      }}
                      className="p-1 text-gray-400 hover:text-blue-500 transition-colors"
                      title="View Document"
                    >
                      <Eye className="w-4 h-4" />
                    </button>

                    <button
                      onClick={() => {
                        const input = document.createElement('input');
                        input.type = 'file';
                        input.accept = allowedTypes.join(',');
                        input.onchange = (e) => {
                          const file = (e.target as HTMLInputElement).files?.[0];
                          if (file) {
                            handleReplaceDocument(doc.id, file, doc.document_type);
                          }
                        };
                        input.click();
                      }}
                      className="p-1 text-gray-400 hover:text-yellow-500 transition-colors"
                      title="Replace Document"
                    >
                      <Upload className="w-4 h-4" />
                    </button>

                    <button
                      onClick={() => handleDeleteDocument(doc.id)}
                      className="p-1 text-gray-400 hover:text-red-500 transition-colors"
                      title="Delete Document"
                    >
                      <X className="w-4 h-4" />
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Document Guidelines */}
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-start">
          <AlertCircle className="w-5 h-5 text-yellow-600 mt-0.5 mr-3 flex-shrink-0" />
          <div>
            <h5 className="font-medium text-yellow-800 mb-2">Document Guidelines</h5>
            <ul className="text-sm text-yellow-700 space-y-1">
              <li>• Ensure all documents are clear and readable</li>
              <li>• Upload original or certified copies when possible</li>
              <li>• Property ownership documents are mandatory for verification</li>
              <li>• All documents should be current and valid</li>
              <li>• Maximum file size: {maxFileSize}MB per document</li>
              <li>• Use password protection (🔒) for sensitive documents</li>
              <li>• Share passwords only with authorized parties</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  );
};