import React, { useEffect, useState } from 'react';
import { useEntitlements } from '../../hooks/useEntitlements';
import { Feature, PackageEntitlement } from '../../types/entitlement';
import { Settings, Plus, ToggleLeft, ToggleRight, Trash2, BarChart3 } from 'lucide-react';

const AdminEntitlementManager: React.FC = () => {
  const {
    getAllFeatures,
    createFeature,
    toggleFeature,
    getPackageEntitlements,
    addPackageEntitlement,
    removePackageEntitlement,
    loading,
    error
  } = useEntitlements();

  const [features, setFeatures] = useState<Feature[]>([]);
  const [packageEntitlements, setPackageEntitlements] = useState<PackageEntitlement[]>([]);
  const [activeTab, setActiveTab] = useState<'features' | 'packages'>('features');
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [newFeature, setNewFeature] = useState({
    code: '',
    name: '',
    category: 'listing',
    description: '',
    requires_subscription: false,
    requires_credits: false,
    credit_cost: 0
  });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    const [featuresData, entitlementsData] = await Promise.all([
      getAllFeatures(),
      getPackageEntitlements()
    ]);
    setFeatures(featuresData);
    setPackageEntitlements(entitlementsData);
  };

  const handleCreateFeature = async () => {
    const success = await createFeature(newFeature);
    if (success) {
      setShowCreateModal(false);
      setNewFeature({
        code: '',
        name: '',
        category: 'listing',
        description: '',
        requires_subscription: false,
        requires_credits: false,
        credit_cost: 0
      });
      loadData();
    }
  };

  const handleToggleFeature = async (code: string) => {
    await toggleFeature(code);
    loadData();
  };

  const handleRemoveEntitlement = async (id: number) => {
    if (confirm('Remove this entitlement?')) {
      await removePackageEntitlement(id);
      loadData();
    }
  };

  const groupedByPackage = packageEntitlements.reduce((acc, item) => {
    const pkg = item.package_name || 'Unknown';
    if (!acc[pkg]) acc[pkg] = [];
    acc[pkg].push(item);
    return acc;
  }, {} as Record<string, PackageEntitlement[]>);

  return (
    <div className="max-w-7xl mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
          <Settings className="w-8 h-8 text-blue-600" />
          Entitlement Management
        </h1>
        <p className="text-gray-600 mt-2">Manage features and package entitlements</p>
      </div>

      {error && (
        <div className="mb-4 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
          {error}
        </div>
      )}

      <div className="mb-6 flex gap-2 border-b border-gray-200">
        <button
          onClick={() => setActiveTab('features')}
          className={`px-6 py-3 font-medium transition-colors ${
            activeTab === 'features'
              ? 'text-blue-600 border-b-2 border-blue-600'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          Features ({features.length})
        </button>
        <button
          onClick={() => setActiveTab('packages')}
          className={`px-6 py-3 font-medium transition-colors ${
            activeTab === 'packages'
              ? 'text-blue-600 border-b-2 border-blue-600'
              : 'text-gray-600 hover:text-gray-900'
          }`}
        >
          Package Entitlements ({packageEntitlements.length})
        </button>
      </div>

      {activeTab === 'features' && (
        <div>
          <div className="mb-4 flex justify-between items-center">
            <h2 className="text-xl font-semibold">All Features</h2>
            <button
              onClick={() => setShowCreateModal(true)}
              className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700"
            >
              <Plus className="w-4 h-4" />
              Create Feature
            </button>
          </div>

          <div className="bg-white rounded-lg shadow overflow-hidden">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Code</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Category</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Type</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Cost</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {features.map((feature) => (
                  <tr key={feature.code} className="hover:bg-gray-50">
                    <td className="px-6 py-4 text-sm font-mono text-gray-900">{feature.code}</td>
                    <td className="px-6 py-4 text-sm text-gray-900">{feature.name}</td>
                    <td className="px-6 py-4 text-sm">
                      <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-xs">
                        {feature.category}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-600">
                      {feature.requires_subscription && <span className="mr-2">📋 Sub</span>}
                      {feature.requires_credits && <span>💳 Credits</span>}
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-900">
                      {feature.credit_cost > 0 ? `${feature.credit_cost} credits` : '-'}
                    </td>
                    <td className="px-6 py-4 text-sm">
                      <span className={`px-2 py-1 rounded-full text-xs ${
                        feature.is_active ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
                      }`}>
                        {feature.is_active ? 'Active' : 'Inactive'}
                      </span>
                    </td>
                    <td className="px-6 py-4 text-sm">
                      <button
                        onClick={() => handleToggleFeature(feature.code)}
                        className="text-blue-600 hover:text-blue-800"
                      >
                        {feature.is_active ? <ToggleRight className="w-5 h-5" /> : <ToggleLeft className="w-5 h-5" />}
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}

      {activeTab === 'packages' && (
        <div className="space-y-6">
          {Object.entries(groupedByPackage).map(([packageName, items]) => (
            <div key={packageName} className="bg-white rounded-lg shadow overflow-hidden">
              <div className="bg-gray-50 px-6 py-4 border-b border-gray-200">
                <h3 className="text-lg font-semibold text-gray-900">{packageName}</h3>
                <p className="text-sm text-gray-600">{items.length} features included</p>
              </div>
              <div className="divide-y divide-gray-200">
                {items.map((item) => (
                  <div key={item.id} className="px-6 py-4 flex items-center justify-between hover:bg-gray-50">
                    <div className="flex-1">
                      <h4 className="font-medium text-gray-900">{item.feature_name}</h4>
                      <div className="flex items-center gap-4 mt-1 text-sm text-gray-600">
                        <span className="px-2 py-1 bg-blue-100 text-blue-800 rounded text-xs">
                          {item.category}
                        </span>
                        {item.limits && Object.keys(item.limits).length > 0 && (
                          <span className="text-gray-500">
                            {item.limits.max_usage && `Limit: ${item.limits.max_usage}`}
                          </span>
                        )}
                      </div>
                    </div>
                    <button
                      onClick={() => item.id && handleRemoveEntitlement(item.id)}
                      className="text-red-600 hover:text-red-800"
                    >
                      <Trash2 className="w-4 h-4" />
                    </button>
                  </div>
                ))}
              </div>
            </div>
          ))}
        </div>
      )}

      {showCreateModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full">
            <h3 className="text-xl font-bold mb-4">Create New Feature</h3>
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Code</label>
                <input
                  type="text"
                  value={newFeature.code}
                  onChange={(e) => setNewFeature({ ...newFeature, code: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                  placeholder="FEATURE_CODE"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Name</label>
                <input
                  type="text"
                  value={newFeature.name}
                  onChange={(e) => setNewFeature({ ...newFeature, name: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Category</label>
                <select
                  value={newFeature.category}
                  onChange={(e) => setNewFeature({ ...newFeature, category: e.target.value })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                >
                  <option value="listing">Listing</option>
                  <option value="lead">Lead</option>
                  <option value="report">Report</option>
                  <option value="addon">Add-on</option>
                  <option value="analytics">Analytics</option>
                  <option value="agent">Agent</option>
                </select>
              </div>
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">Credit Cost</label>
                <input
                  type="number"
                  value={newFeature.credit_cost}
                  onChange={(e) => setNewFeature({ ...newFeature, credit_cost: parseInt(e.target.value) })}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg"
                />
              </div>
              <div className="flex items-center gap-4">
                <label className="flex items-center gap-2">
                  <input
                    type="checkbox"
                    checked={newFeature.requires_subscription}
                    onChange={(e) => setNewFeature({ ...newFeature, requires_subscription: e.target.checked })}
                    className="rounded"
                  />
                  <span className="text-sm">Requires Subscription</span>
                </label>
                <label className="flex items-center gap-2">
                  <input
                    type="checkbox"
                    checked={newFeature.requires_credits}
                    onChange={(e) => setNewFeature({ ...newFeature, requires_credits: e.target.checked })}
                    className="rounded"
                  />
                  <span className="text-sm">Requires Credits</span>
                </label>
              </div>
            </div>
            <div className="mt-6 flex gap-3">
              <button
                onClick={handleCreateFeature}
                disabled={loading}
                className="flex-1 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
              >
                Create
              </button>
              <button
                onClick={() => setShowCreateModal(false)}
                className="flex-1 px-4 py-2 bg-gray-200 text-gray-800 rounded-lg hover:bg-gray-300"
              >
                Cancel
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default AdminEntitlementManager;
