import React, { useEffect, useState } from 'react';
import { useEntitlements } from '../../hooks/useEntitlements';
import { EntitlementCheckResponse } from '../../types/entitlement';
import { Lock, CheckCircle, AlertCircle } from 'lucide-react';

interface Props {
  userId: number;
  userType: string;
  featureCode: string;
  onGranted?: () => void;
  onDenied?: (reason: string) => void;
  children?: React.ReactNode;
  showMessage?: boolean;
}

const EntitlementGate: React.FC<Props> = ({
  userId,
  userType,
  featureCode,
  onGranted,
  onDenied,
  children,
  showMessage = true
}) => {
  const { checkEntitlement, useFeature, loading } = useEntitlements();
  const [entitlement, setEntitlement] = useState<EntitlementCheckResponse | null>(null);
  const [checked, setChecked] = useState(false);

  useEffect(() => {
    checkAccess();
  }, [userId, userType, featureCode]);

  const checkAccess = async () => {
    const result = await checkEntitlement({
      user_id: userId,
      user_type: userType,
      feature_code: featureCode
    });
    
    if (result) {
      setEntitlement(result);
      setChecked(true);
      
      if (result.is_granted && onGranted) {
        onGranted();
      } else if (!result.is_granted && onDenied) {
        onDenied(result.denial_reason || 'Access denied');
      }
    }
  };

  const handleUseFeature = async () => {
    const result = await useFeature({
      user_id: userId,
      user_type: userType,
      feature_code: featureCode
    });
    
    if (result?.success) {
      checkAccess();
    }
  };

  if (loading || !checked) {
    return (
      <div className="flex items-center justify-center p-4">
        <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (!entitlement?.is_granted) {
    return (
      <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
        <div className="flex items-start gap-3">
          <Lock className="w-5 h-5 text-yellow-600 flex-shrink-0 mt-0.5" />
          <div className="flex-1">
            <h4 className="font-semibold text-gray-900 mb-1">Access Restricted</h4>
            {showMessage && (
              <p className="text-sm text-gray-600 mb-3">{entitlement?.denial_reason}</p>
            )}
            {entitlement?.credit_cost > 0 && (
              <div className="text-sm text-gray-700 mb-3">
                <span className="font-medium">Cost:</span> {entitlement.credit_cost} credits
                {entitlement.current_balance !== undefined && (
                  <span className="ml-2 text-gray-500">
                    (Your balance: {entitlement.current_balance})
                  </span>
                )}
              </div>
            )}
            <div className="flex gap-2">
              <button className="px-3 py-1.5 bg-blue-600 text-white text-sm rounded hover:bg-blue-700">
                Upgrade Plan
              </button>
              {entitlement?.credit_cost > 0 && (
                <button className="px-3 py-1.5 bg-green-600 text-white text-sm rounded hover:bg-green-700">
                  Buy Credits
                </button>
              )}
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div>
      {showMessage && (
        <div className="bg-green-50 border border-green-200 rounded-lg p-3 mb-4">
          <div className="flex items-center gap-2 text-sm text-green-800">
            <CheckCircle className="w-4 h-4" />
            <span>Access granted via {entitlement.entitlement_source}</span>
            {entitlement.credit_cost > 0 && (
              <span className="ml-auto font-medium">{entitlement.credit_cost} credits</span>
            )}
          </div>
        </div>
      )}
      {children}
    </div>
  );
};

export default EntitlementGate;
