import React, { useEffect, useState } from 'react';
import { useEntitlements } from '../../hooks/useEntitlements';
import { UserEntitlement } from '../../types/entitlement';
import { Shield, Lock, CreditCard, CheckCircle } from 'lucide-react';

interface Props {
  userId: number;
  userType: string;
}

const UserEntitlementsDashboard: React.FC<Props> = ({ userId, userType }) => {
  const { getMyEntitlements, loading } = useEntitlements();
  const [entitlements, setEntitlements] = useState<UserEntitlement[]>([]);
  const [groupedEntitlements, setGroupedEntitlements] = useState<Record<string, UserEntitlement[]>>({});

  useEffect(() => {
    loadEntitlements();
  }, [userId, userType]);

  const loadEntitlements = async () => {
    const data = await getMyEntitlements(userId, userType);
    setEntitlements(data);
    
    const grouped = data.reduce((acc, item) => {
      if (!acc[item.category]) acc[item.category] = [];
      acc[item.category].push(item);
      return acc;
    }, {} as Record<string, UserEntitlement[]>);
    
    setGroupedEntitlements(grouped);
  };

  const getSourceBadge = (source: string) => {
    const colors = {
      subscription: 'bg-blue-100 text-blue-800',
      credits: 'bg-green-100 text-green-800',
      admin_grant: 'bg-purple-100 text-purple-800',
      free: 'bg-gray-100 text-gray-800'
    };
    return colors[source as keyof typeof colors] || colors.free;
  };

  const getCategoryIcon = (category: string) => {
    const icons: Record<string, string> = {
      listing: '🏠', lead: '👥', report: '📊', addon: '⭐', analytics: '📈', agent: '💼'
    };
    return icons[category] || '📦';
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center p-8">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="max-w-6xl mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-gray-900 flex items-center gap-3">
          <Shield className="w-8 h-8 text-blue-600" />
          My Entitlements
        </h1>
        <p className="text-gray-600 mt-2">Features and services you have access to</p>
      </div>

      {entitlements.length === 0 ? (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-6 text-center">
          <Lock className="w-12 h-12 text-yellow-600 mx-auto mb-3" />
          <h3 className="text-lg font-semibold text-gray-900 mb-2">No Active Entitlements</h3>
          <p className="text-gray-600">Subscribe to a plan or purchase credits to access features</p>
        </div>
      ) : (
        <div className="space-y-6">
          {Object.entries(groupedEntitlements).map(([category, items]) => (
            <div key={category} className="bg-white rounded-lg shadow-md overflow-hidden">
              <div className="bg-gradient-to-r from-blue-600 to-blue-700 px-6 py-4">
                <h2 className="text-xl font-semibold text-white flex items-center gap-2">
                  <span>{getCategoryIcon(category)}</span>
                  {category.charAt(0).toUpperCase() + category.slice(1)} Features
                  <span className="ml-auto bg-white/20 px-3 py-1 rounded-full text-sm">
                    {items.length}
                  </span>
                </h2>
              </div>
              
              <div className="divide-y divide-gray-200">
                {items.map((item, idx) => (
                  <div key={idx} className="p-4 hover:bg-gray-50 transition-colors">
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <div className="flex items-center gap-3">
                          <CheckCircle className="w-5 h-5 text-green-600 flex-shrink-0" />
                          <h3 className="font-semibold text-gray-900">{item.name}</h3>
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${getSourceBadge(item.source)}`}>
                            {item.source}
                          </span>
                        </div>
                        
                        <div className="mt-2 ml-8 flex items-center gap-4 text-sm text-gray-600">
                          {item.credit_cost !== undefined && item.credit_cost > 0 && (
                            <span className="flex items-center gap-1">
                              <CreditCard className="w-4 h-4" />
                              {item.credit_cost} credits
                            </span>
                          )}
                          
                          {item.limits && Object.keys(item.limits).length > 0 && (
                            <span className="text-gray-500">
                              {item.limits.max_usage && `Limit: ${item.limits.max_usage}`}
                              {item.limits.period && ` per ${item.limits.period}`}
                            </span>
                          )}
                          
                          {item.current_balance !== undefined && (
                            <span className="text-green-600 font-medium">
                              Balance: {item.current_balance} credits
                            </span>
                          )}
                        </div>
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            </div>
          ))}
        </div>
      )}

      <div className="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-6">
        <h3 className="font-semibold text-gray-900 mb-2">Need More Features?</h3>
        <p className="text-gray-600 mb-4">Upgrade your plan or purchase credits to unlock additional features</p>
        <div className="flex gap-3">
          <button className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors">
            View Plans
          </button>
          <button className="px-4 py-2 bg-white border border-blue-600 text-blue-600 rounded-lg hover:bg-blue-50 transition-colors">
            Buy Credits
          </button>
        </div>
      </div>
    </div>
  );
};

export default UserEntitlementsDashboard;
