import React, { useState, useEffect } from 'react';
import { escrowApi } from '@/api/escrow';
import { Button } from '@/components/ui/button';
import { Card } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { toast } from '@/utils/toast';
import { PaymentModal } from '@/components/transactions/PaymentModal';
import { TransactionType } from '@/types/transaction';

export const EscrowDashboard: React.FC = () => {
  const [escrows, setEscrows] = useState<any[]>([]);
  const [selectedEscrow, setSelectedEscrow] = useState<any>(null);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [loading, setLoading] = useState(true);
  const user = JSON.parse(localStorage.getItem('user') || '{}');

  useEffect(() => {
    loadEscrows();
  }, []);

  const loadEscrows = async () => {
    try {
      setLoading(true);
      const data = await escrowApi.list();
      setEscrows(Array.isArray(data) ? data : []);
    } catch (error) {
      console.error('Error loading escrows:', error);
      setEscrows([]);
    } finally {
      setLoading(false);
    }
  };

  const handleDeposit = async (transactionId: string) => {
    if (!selectedEscrow) return;

    try {
      await escrowApi.deposit(selectedEscrow.id, {
        amount: selectedEscrow.escrow_amount,
        payment_method: 'upi',
        transaction_id: transactionId
      });

      toast.success('Funds deposited to escrow successfully!');
      loadEscrows();
    } catch (error) {
      toast.error('Failed to deposit funds');
    }
  };

  const getStatusColor = (status: string) => {
    const colors: any = {
      pending: 'bg-yellow-100 text-yellow-800',
      funded: 'bg-green-100 text-green-800',
      held: 'bg-blue-100 text-blue-800',
      released: 'bg-purple-100 text-purple-800',
      refunded: 'bg-red-100 text-red-800',
      cancelled: 'bg-gray-100 text-gray-800'
    };
    return colors[status] || 'bg-gray-100 text-gray-800';
  };

  const getStatusLabel = (status: string) => {
    return status === 'refunded' ? 'TERMINATED' : status.toUpperCase();
  };

  if (loading) return <div>Loading...</div>;

  return (
    <div className="space-y-6">
      <h2 className="text-2xl font-bold">Escrow Accounts</h2>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
        {escrows.map((escrow) => (
          <Card key={escrow.id} className="p-6">
            <div className="flex justify-between items-start mb-4">
              <div>
                <h3 className="text-lg font-semibold">{escrow.escrow_number}</h3>
                <p className="text-sm text-gray-600">Property #{escrow.property_id}</p>
              </div>
              <Badge className={getStatusColor(escrow.status)}>
                {getStatusLabel(escrow.status)}
              </Badge>
            </div>

            <div className="space-y-2 mb-4">
              <div className="flex justify-between">
                <span className="text-gray-600">Escrow Amount:</span>
                <span className="font-semibold">₹{escrow.escrow_amount?.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Current Balance:</span>
                <span className="font-semibold">₹{escrow.current_balance?.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Conditions Met:</span>
                <span className={escrow.conditions_met ? 'text-green-600' : 'text-red-600'}>
                  {escrow.conditions_met ? 'Yes' : 'No'}
                </span>
              </div>
            </div>

            {escrow.status === 'pending' && escrow.buyer_id === user.user_id && (
              <Button
                onClick={() => {
                  setSelectedEscrow(escrow);
                  setShowPaymentModal(true);
                }}
                className="w-full"
              >
                Deposit Funds
              </Button>
            )}

            {escrow.status === 'pending' && escrow.seller_id === user.user_id && (
              <div className="text-sm text-gray-600">
                Waiting for buyer to deposit funds
              </div>
            )}

            {escrow.status === 'funded' && (
              <div className="text-sm text-gray-600">
                Waiting for conditions to be met
              </div>
            )}

            {escrow.status === 'held' && (
              <div className="text-sm text-green-600">
                Ready for release
              </div>
            )}

            {escrow.status === 'refunded' && (
              <div className="bg-red-50 border border-red-200 rounded p-3 text-center">
                <p className="text-red-800 font-semibold">⚠️ TERMINATED</p>
                <p className="text-xs text-red-600 mt-1">This escrow account has been refunded and closed permanently</p>
              </div>
            )}
          </Card>
        ))}
      </div>

      {escrows.length === 0 && (
        <Card className="p-8 text-center">
          <svg className="w-16 h-16 text-gray-400 mx-auto mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z" />
          </svg>
          <h3 className="text-lg font-semibold text-gray-900 mb-2">No Escrow Accounts Yet</h3>
          <p className="text-gray-600 mb-4">
            Escrow accounts are automatically created when both parties sign a contract.
          </p>
          <p className="text-sm text-gray-500">
            Complete the agreement → contract → signing process to create an escrow account.
          </p>
        </Card>
      )}

      {showPaymentModal && selectedEscrow && (
        <PaymentModal
          isOpen={showPaymentModal}
          onClose={() => setShowPaymentModal(false)}
          amount={selectedEscrow.escrow_amount}
          propertyId={selectedEscrow.property_id}
          transactionType={TransactionType.ESCROW_DEPOSIT}
          description={`Escrow deposit for ${selectedEscrow.escrow_number}`}
          onSuccess={handleDeposit}
        />
      )}
    </div>
  );
};
