import React, { useEffect, useState } from 'react';
import { useVerificationFeatures } from '../../hooks/useVerificationFeatures';
import ValidationResults from './ValidationResults';
import ForgeryDetection from './ForgeryDetection';
import DuplicateDetection from './DuplicateDetection';
import AuthenticityScore from './AuthenticityScore';
import DisputeRisk from './DisputeRisk';
import { DuplicateWarningModal } from './Modals';
import { ErrorAlert, SuccessAlert } from './Alerts';
import LoadingSkeleton from './LoadingSkeleton';

interface VerificationIntegrationProps {
  documentId: string;
  onVerificationComplete?: (result: any) => void;
}

export const VerificationIntegration: React.FC<VerificationIntegrationProps> = ({
  documentId,
  onVerificationComplete,
}) => {
  const {
    verification,
    feature1,
    feature2,
    feature3,
    feature4,
    feature5,
    isLoading,
    error,
    fetchVerificationSummary,
    clearError,
    reset,
  } = useVerificationFeatures();

  const [showDuplicateModal, setShowDuplicateModal] = useState(false);
  const [showSuccess, setShowSuccess] = useState(false);

  useEffect(() => {
    if (documentId) {
      fetchVerificationSummary(documentId);
    }
  }, [documentId, fetchVerificationSummary]);

  useEffect(() => {
    if (feature3?.is_duplicate) {
      setShowDuplicateModal(true);
    }
  }, [feature3]);

  useEffect(() => {
    if (verification && !isLoading && !error) {
      setShowSuccess(true);
      if (onVerificationComplete) {
        onVerificationComplete(verification);
      }
    }
  }, [verification, isLoading, error, onVerificationComplete]);

  if (isLoading && !verification) {
    return (
      <div className="space-y-6">
        <LoadingSkeleton type="card" count={5} />
      </div>
    );
  }

  return (
    <div className="space-y-6">
      {error && (
        <ErrorAlert
          message={error.message}
          details={error.error_code}
          onClose={clearError}
        />
      )}

      {showSuccess && (
        <SuccessAlert
          message="Document verification completed successfully"
          onClose={() => setShowSuccess(false)}
        />
      )}

      {feature1 && (
        <ValidationResults
          data={feature1}
          isLoading={isLoading}
          error={error?.message || null}
        />
      )}

      {feature2 && (
        <ForgeryDetection
          data={feature2}
          isLoading={isLoading}
          error={error?.message || null}
        />
      )}

      {feature3 && (
        <DuplicateDetection
          data={feature3}
          isLoading={isLoading}
          error={error?.message || null}
        />
      )}

      {feature4 && (
        <AuthenticityScore
          data={feature4}
          isLoading={isLoading}
          error={error?.message || null}
        />
      )}

      {feature5 && (
        <DisputeRisk
          data={feature5}
          isLoading={isLoading}
          error={error?.message || null}
        />
      )}

      <DuplicateWarningModal
        isOpen={showDuplicateModal}
        data={feature3 || null}
        onContinue={() => setShowDuplicateModal(false)}
        onCancel={() => {
          setShowDuplicateModal(false);
          reset();
        }}
      />
    </div>
  );
};

export default VerificationIntegration;
