import React from 'react';
import { Download, FileText, ChevronRight, Eye } from 'lucide-react';

interface Invoice {
    id: string;
    number: string;
    date: string;
    amount: number;
    status: 'paid' | 'pending' | 'refunded';
    pdf_url: string;
}

interface InvoiceListProps {
    invoices: Invoice[];
    onDownload: (id: string) => void;
    onView: (id: string) => void;
}

const InvoiceList: React.FC<InvoiceListProps> = ({ invoices, onDownload, onView }) => {
    if (invoices.length === 0) {
        return (
            <div className="bg-white rounded-xl shadow-sm border border-gray-100 p-12 text-center">
                <div className="mx-auto w-12 h-12 bg-gray-100 rounded-full flex items-center justify-center mb-4">
                    <FileText className="w-6 h-6 text-gray-400" />
                </div>
                <h3 className="text-lg font-medium text-gray-900">No invoices yet</h3>
                <p className="text-gray-500 mt-1">You haven't made any purchases yet.</p>
            </div>
        );
    }

    return (
        <div className="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
            <div className="px-6 py-4 border-b border-gray-100 bg-gray-50 flex justify-between items-center">
                <h3 className="text-lg font-semibold text-gray-900 block">Recent Invoices</h3>
            </div>
            <div className="divide-y divide-gray-100">
                {invoices.map((invoice) => (
                    <div key={invoice.id} className="p-4 hover:bg-gray-50 transition-colors flex items-center justify-between group">
                        <div className="flex items-center gap-4">
                            <div className="p-2 bg-blue-50 text-[#0056D2] rounded-lg">
                                <FileText className="w-5 h-5" />
                            </div>
                            <div>
                                <h4 className="font-medium text-gray-900">{invoice.number}</h4>
                                <p className="text-sm text-gray-500">{new Date(invoice.date).toLocaleDateString()}</p>
                            </div>
                        </div>

                        <div className="flex items-center gap-6">
                            <div className="text-right">
                                <p className="font-semibold text-gray-900">${invoice.amount.toFixed(2)}</p>
                                <span className={`text-xs font-medium px-2 py-0.5 rounded-full capitalize
                        ${invoice.status === 'paid' ? 'bg-green-100 text-[#00BFA6]' :
                                        invoice.status === 'refunded' ? 'bg-gray-100 text-gray-600' : 'bg-yellow-100 text-yellow-600'}`
                                }>
                                    {invoice.status}
                                </span>
                            </div>

                            <div className="flex items-center gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                <button
                                    onClick={() => onDownload(invoice.id)}
                                    className="p-2 text-gray-400 hover:text-[#0056D2] hover:bg-blue-50 rounded-full transition-colors"
                                    title="Download PDF"
                                >
                                    <Download className="w-5 h-5" />
                                </button>
                                <button
                                    onClick={() => onView(invoice.id)}
                                    className="p-2 text-gray-400 hover:text-[#0056D2] hover:bg-blue-50 rounded-full transition-colors"
                                    title="View Details"
                                >
                                    <Eye className="w-5 h-5" />
                                </button>
                            </div>
                        </div>
                    </div>
                ))}
            </div>
            <div className="px-6 py-3 bg-gray-50 border-t border-gray-100 text-center">
                <button className="text-sm font-medium text-[#0056D2] hover:text-[#0041A3] flex items-center justify-center gap-1 mx-auto">
                    View All Invoices <ChevronRight className="w-4 h-4" />
                </button>
            </div>
        </div>
    );
};

export default InvoiceList;
