import React, { useState } from 'react';
import { X, AlertCircle } from 'lucide-react';

interface RefundRequestModalProps {
    invoiceId: string;
    isOpen: boolean;
    onClose: () => void;
    onSubmit: (reason: string, details: string) => void;
}

const RefundRequestModal: React.FC<RefundRequestModalProps> = ({
    invoiceId,
    isOpen,
    onClose,
    onSubmit
}) => {
    const [reason, setReason] = useState('accidental');
    const [details, setDetails] = useState('');

    if (!isOpen) return null;

    const handleSubmit = (e: React.FormEvent) => {
        e.preventDefault();
        onSubmit(reason, details);
        onClose();
    };

    return (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50 backdrop-blur-sm">
            <div className="bg-white rounded-xl shadow-xl w-full max-w-md overflow-hidden">
                <div className="px-6 py-4 border-b border-gray-100 flex justify-between items-center bg-gray-50">
                    <h3 className="text-lg font-semibold text-gray-900">Request Refund</h3>
                    <button onClick={onClose} className="text-gray-400 hover:text-gray-600 transition-colors">
                        <X className="w-5 h-5" />
                    </button>
                </div>

                <div className="p-6">
                    <div className="mb-6 flex items-start gap-3 p-4 bg-amber-50 rounded-lg text-amber-800 text-sm">
                        <AlertCircle className="w-5 h-5 flex-shrink-0 mt-0.5" />
                        <p>
                            Refunds for Invoice <strong>#{invoiceId}</strong> are processed within 5-7 business days.
                            The amount will be credited to your original payment method.
                        </p>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-4">
                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                Reason for Refund
                            </label>
                            <select
                                value={reason}
                                onChange={(e) => setReason(e.target.value)}
                                className="w-full rounded-lg border-gray-300 shadow-sm focus:border-[#0056D2] focus:ring-[#0056D2] sm:text-sm py-2.5 px-3 border"
                            >
                                <option value="accidental">Accidental Purchase</option>
                                <option value="fraud">Unrecognized Charge</option>
                                <option value="duplicate">Duplicate Charge</option>
                                <option value="unsatisfied">Service Not as Described</option>
                                <option value="other">Other</option>
                            </select>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">
                                Additional Details
                            </label>
                            <textarea
                                value={details}
                                onChange={(e) => setDetails(e.target.value)}
                                rows={3}
                                className="w-full rounded-lg border-gray-300 shadow-sm focus:border-[#0056D2] focus:ring-[#0056D2] sm:text-sm py-2 px-3 border"
                                placeholder="Please explain why you are requesting a refund..."
                                required
                            />
                        </div>

                        <div className="pt-4 flex gap-3">
                            <button
                                type="button"
                                onClick={onClose}
                                className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 font-medium rounded-lg hover:bg-gray-50 transition-colors"
                            >
                                Cancel
                            </button>
                            <button
                                type="submit"
                                className="flex-1 px-4 py-2 bg-[#0056D2] text-white font-medium rounded-lg hover:bg-[#0041A3] transition-colors shadow-sm"
                            >
                                Submit Request
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    );
};

export default RefundRequestModal;
