import React from 'react';
import { Edit2, Trash2, Plus, ToggleLeft, ToggleRight } from 'lucide-react';

interface TaxRate {
    id: string;
    region: string;
    name: string;
    percentage: number;
    is_active: boolean;
}

interface TaxSettingsTableProps {
    rates: TaxRate[];
    onEdit: (rate: TaxRate) => void;
    onDelete: (id: string) => void;
    onToggle: (id: string) => void;
    onAdd: () => void;
}

const TaxSettingsTable: React.FC<TaxSettingsTableProps> = ({ rates, onEdit, onDelete, onToggle, onAdd }) => {
    return (
        <div className="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
            <div className="px-6 py-4 border-b border-gray-100 flex justify-between items-center bg-gray-50">
                <h3 className="text-lg font-semibold text-gray-900">Tax Rates Configuration</h3>
                <button
                    onClick={onAdd}
                    className="flex items-center gap-2 px-4 py-2 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg text-sm font-medium transition-colors"
                >
                    <Plus className="w-4 h-4" /> Add Tax Rate
                </button>
            </div>

            <div className="overflow-x-auto">
                <table className="w-full text-left text-sm">
                    <thead className="bg-gray-50 text-gray-500 uppercase font-medium">
                        <tr>
                            <th className="px-6 py-3">Region Code</th>
                            <th className="px-6 py-3">Tax Name</th>
                            <th className="px-6 py-3">Rate (%)</th>
                            <th className="px-6 py-3">Status</th>
                            <th className="px-6 py-3 text-right">Actions</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {rates.map((rate) => (
                            <tr key={rate.id} className="hover:bg-gray-50 group">
                                <td className="px-6 py-4 font-medium text-gray-900">{rate.region}</td>
                                <td className="px-6 py-4 text-gray-600">{rate.name}</td>
                                <td className="px-6 py-4 font-semibold text-gray-900">{rate.percentage.toFixed(2)}%</td>
                                <td className="px-6 py-4">
                                    <button
                                        onClick={() => onToggle(rate.id)}
                                        className={`flex items-center gap-1 transition-colors ${rate.is_active ? 'text-[#00BFA6]' : 'text-gray-400'}`}
                                    >
                                        {rate.is_active ? <ToggleRight className="w-6 h-6" /> : <ToggleLeft className="w-6 h-6" />}
                                        <span className="text-xs font-medium">{rate.is_active ? 'Active' : 'Inactive'}</span>
                                    </button>
                                </td>
                                <td className="px-6 py-4 text-right flex justify-end gap-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                    <button onClick={() => onEdit(rate)} className="p-1.5 text-gray-400 hover:text-[#0056D2] hover:bg-blue-50 rounded transistion-colors">
                                        <Edit2 className="w-4 h-4" />
                                    </button>
                                    <button onClick={() => onDelete(rate.id)} className="p-1.5 text-gray-400 hover:text-red-600 hover:bg-red-50 rounded transistion-colors">
                                        <Trash2 className="w-4 h-4" />
                                    </button>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
                {rates.length === 0 && (
                    <div className="p-8 text-center text-gray-400">
                        No tax rates configured. Click "Add Tax Rate" to start.
                    </div>
                )}
            </div>
        </div>
    );
};

export default TaxSettingsTable;
