import React from 'react';
import { Activity, AlertTriangle, ArrowRight } from 'lucide-react';

interface UsageStats {
    feature_slug: string;
    count: number;
    limit: number;
    reset_at: string;
}

interface UsageMeterProps {
    stats: UsageStats;
    onUpgrade: () => void;
}

const UsageMeter: React.FC<UsageMeterProps> = ({ stats, onUpgrade }) => {
    const percentage = Math.min((stats.count / stats.limit) * 100, 100);
    const isCritical = percentage >= 90;
    const isWarning = percentage >= 75 && percentage < 90;

    // NAL Colors
    const barColor = isCritical ? 'bg-red-500' : isWarning ? 'bg-[#00BFA6]' : 'bg-[#0056D2]';
    const textColor = isCritical ? 'text-red-600' : 'text-[#0056D2]';

    return (
        <div className="bg-white rounded-lg shadow-sm border border-gray-100 p-6 flex flex-col gap-4">
            <div className="flex justify-between items-center">
                <div className="flex items-center gap-3">
                    <div className={`p-2 rounded-full ${isCritical ? 'bg-red-50' : 'bg-blue-50'}`}>
                        <Activity className={`w-5 h-5 ${textColor}`} />
                    </div>
                    <div>
                        <h3 className="text-lg font-semibold text-gray-900 capitalize">
                            {stats.feature_slug.replace('_', ' ')}
                        </h3>
                        <p className="text-sm text-gray-500">
                            Resets on {new Date(stats.reset_at).toLocaleDateString()}
                        </p>
                    </div>
                </div>

                {isCritical && (
                    <div className="flex items-center gap-1 text-red-600 bg-red-50 px-3 py-1 rounded-full text-xs font-medium">
                        <AlertTriangle className="w-3 h-3" />
                        <span>Limit Reached</span>
                    </div>
                )}
            </div>

            <div className="space-y-2">
                <div className="flex justify-between text-sm font-medium">
                    <span className="text-gray-700">{stats.count.toLocaleString()} Used</span>
                    <span className="text-gray-500">{stats.limit.toLocaleString()} Limit</span>
                </div>
                <div className="w-full bg-gray-100 rounded-full h-3 overflow-hidden">
                    <div
                        className={`h-full rounded-full transition-all duration-500 ease-out ${barColor}`}
                        style={{ width: `${percentage}%` }}
                    />
                </div>
            </div>

            <div className="pt-2">
                <button
                    onClick={onUpgrade}
                    className="w-full flex items-center justify-center gap-2 py-2.5 px-4 bg-[#0056D2] hover:bg-[#0041A3] text-white rounded-lg font-medium transition-colors text-sm"
                >
                    Upgrade Plan <ArrowRight className="w-4 h-4" />
                </button>
            </div>
        </div>
    );
};

export default UsageMeter;
