import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';

const categories = [
  'Residential Apartment',
  'Independent House',
  'Villa',
  'Commercial Office',
  'Retail Space',
  'Warehouse',
  'Land/Plot'
];

export const BasicInfoStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger } = useFormContext<ListingFormData>();

  const handleNext = async () => {
    const isValid = await trigger(['title', 'category', 'description']);
    if (isValid) onNext();
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Basic Information</h2>
        <p className="text-gray-600">Tell us about your property</p>
      </div>

      <div className="space-y-6">
        {/* Property Title */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property Title *
          </label>
          <input
            type="text"
            {...register('title', validation.required('Property title is required'))}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
            placeholder="e.g., Luxury 3BHK Apartment in Prime Location"
          />
          {errors.title && (
            <p className="mt-1 text-sm text-red-600">{errors.title.message}</p>
          )}
        </div>

        {/* Category */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property Category *
          </label>
          <select
            {...register('category', validation.required('Please select a category'))}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
          >
            <option value="">Select Category</option>
            {categories.map(category => (
              <option key={category} value={category}>{category}</option>
            ))}
          </select>
          {errors.category && (
            <p className="mt-1 text-sm text-red-600">{errors.category.message}</p>
          )}
        </div>

        {/* Description */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property Description *
          </label>
          <textarea
            {...register('description', {
              ...validation.required('Description is required'),
              ...validation.minLength(50, 'Description must be at least 50 characters')
            })}
            rows={6}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent resize-none"
            placeholder="Describe your property in detail. Include key features, amenities, and what makes it special..."
          />
          {errors.description && (
            <p className="mt-1 text-sm text-red-600">{errors.description.message}</p>
          )}
          <p className="mt-1 text-sm text-gray-500">Minimum 50 characters required</p>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between pt-6">
        <div>
          {!isFirst && (
            <button
              type="button"
              onClick={onPrev}
              className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
            >
              Previous
            </button>
          )}
        </div>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Details
        </button>
      </div>
    </motion.div>
  );
};