import React from 'react';
import { useFormContext } from 'react-hook-form';

interface ControlledInputProps {
  name: string;
  label: string;
  type?: string;
  placeholder?: string;
  required?: boolean;
}

export const ControlledInputExample: React.FC<ControlledInputProps> = ({
  name,
  label,
  type = 'text',
  placeholder,
  required = false
}) => {
  const { setValue, watch, formState: { errors } } = useFormContext();
  
  // Get current value - this is stable and won't cause re-renders
  const value = watch(name) || '';
  
  // Stable onChange handler
  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setValue(name, e.target.value, { 
      shouldValidate: true,
      shouldDirty: true 
    });
  };

  const error = errors[name];

  return (
    <div>
      <label className="block text-sm font-medium text-gray-700 mb-2">
        {label} {required && <span className="text-red-500">*</span>}
      </label>
      <input
        type={type}
        name={name}
        value={value}
        onChange={handleChange}
        className="w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white"
        placeholder={placeholder}
      />
      {error && (
        <p className="mt-1 text-sm text-red-500">{error.message}</p>
      )}
    </div>
  );
};

// Example usage in your form:
/*
<ControlledInputExample
  name="title"
  label="Property Title"
  placeholder="e.g., Luxury 3BHK Apartment"
  required={true}
/>
*/