import React, { useState } from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { Plus, X } from 'lucide-react';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';

const commonSpecifications = [
  'Air Conditioning',
  'Parking Space',
  'Balcony',
  'Furnished',
  'Semi-Furnished',
  'Unfurnished',
  'Elevator',
  'Security',
  'Power Backup',
  'Water Supply'
];

const commonAmenities = [
  'Swimming Pool',
  'Gym',
  'Parking',
  'Security',
  'Garden',
  'Elevator',
  'Power Backup',
  'Water Supply',
  'Club House',
  'Children Play Area',
  'Jogging Track',
  'CCTV'
];

const propertyHighlights = [
  'Prime Location',
  'Vastu Compliant',
  'Corner Property',
  'Park Facing',
  'Gated Community',
  'Near Metro',
  'Investment Property',
  'Ready to Move'
];

export const DetailsStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger, watch, setValue } = useFormContext<ListingFormData>();
  const [customFeature, setCustomFeature] = useState('');
  const [customAmenity, setCustomAmenity] = useState('');
  const specifications = watch('specifications') || [];
  const features = watch('features') || [];
  const amenities = watch('amenities') || [];
  const highlights = watch('highlights') || [];

  const handleSpecificationChange = (spec: string, checked: boolean) => {
    const current = specifications || [];
    if (checked) {
      setValue('specifications', [...current, spec]);
    } else {
      setValue('specifications', current.filter(s => s !== spec));
    }
  };

  const handleAmenityChange = (amenity: string, checked: boolean) => {
    const current = amenities || [];
    if (checked) {
      setValue('amenities', [...current, amenity]);
    } else {
      setValue('amenities', current.filter(a => a !== amenity));
    }
  };

  const handleHighlightChange = (highlight: string, checked: boolean) => {
    const current = highlights || [];
    if (checked) {
      setValue('highlights', [...current, highlight]);
    } else {
      setValue('highlights', current.filter(h => h !== highlight));
    }
  };

  const addCustomFeature = () => {
    if (customFeature.trim()) {
      const current = features || [];
      setValue('features', [...current, customFeature.trim()]);
      setCustomFeature('');
    }
  };

  const addCustomAmenity = () => {
    if (customAmenity.trim()) {
      const current = amenities || [];
      setValue('amenities', [...current, customAmenity.trim()]);
      setCustomAmenity('');
    }
  };

  const removeFeature = (index: number) => {
    const current = features || [];
    setValue('features', current.filter((_, i) => i !== index));
  };

  const handleNext = async () => {
    // No validation required for this step - it's optional details
    onNext();
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Details</h2>
        <p className="text-gray-600">Add pricing and specifications</p>
      </div>

      <div className="space-y-6">
        {/* Detailed Pricing */}
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Detailed Pricing</h3>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {/* Property Price */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Property Price (₹) *
              </label>
              <input
                type="number"
                {...register('price', {
                  ...validation.required('Property price is required'),
                  ...validation.min(1, 'Price must be greater than 0')
                })}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                placeholder="e.g., 8500000"
              />
              {errors.price && (
                <p className="mt-1 text-sm text-red-600">{errors.price.message}</p>
              )}
            </div>

            {/* Price per Sq Ft */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Price per Sq Ft (₹)
              </label>
              <input
                type="number"
                {...register('pricePerSqFt')}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
                placeholder="e.g., 5500"
              />
            </div>
          </div>
        </div>

        {/* Property Specifications */}
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Specifications</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
            {commonSpecifications.map(spec => (
              <label key={spec} className="flex items-center space-x-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={specifications.includes(spec)}
                  onChange={(e) => handleSpecificationChange(spec, e.target.checked)}
                  className="w-4 h-4 text-[#2B256D] border-gray-300 rounded focus:ring-[#2B256D]"
                />
                <span className="text-sm text-gray-700">{spec}</span>
              </label>
            ))}
          </div>
        </div>

        {/* Additional Features */}
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Additional Features</h3>
          
          {/* Add Feature Input */}
          <div className="flex gap-2 mb-4">
            <input
              type="text"
              value={customFeature}
              onChange={(e) => setCustomFeature(e.target.value)}
              placeholder="Add custom feature..."
              className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              onKeyPress={(e) => e.key === 'Enter' && addCustomFeature()}
            />
            <button
              type="button"
              onClick={addCustomFeature}
              className="px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#1f1a5a] transition-colors"
            >
              <Plus className="w-4 h-4" />
            </button>
          </div>

          {/* Feature Tags */}
          {features.length > 0 && (
            <div className="flex flex-wrap gap-2">
              {features.map((feature, index) => (
                <span
                  key={index}
                  className="inline-flex items-center px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm"
                >
                  {feature}
                  <button
                    type="button"
                    onClick={() => removeFeature(index)}
                    className="ml-2 text-blue-600 hover:text-blue-800"
                  >
                    <X className="w-3 h-3" />
                  </button>
                </span>
              ))}
            </div>
          )}
        </div>

        {/* Amenities & Features */}
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Amenities & Features</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-3 mb-4">
            {commonAmenities.map(amenity => (
              <label key={amenity} className="flex items-center space-x-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={amenities.includes(amenity)}
                  onChange={(e) => handleAmenityChange(amenity, e.target.checked)}
                  className="w-4 h-4 text-[#2B256D] border-gray-300 rounded focus:ring-[#2B256D]"
                />
                <span className="text-sm text-gray-700">{amenity}</span>
              </label>
            ))}
          </div>
          
          {/* Add Custom Amenity */}
          <div className="flex gap-2 mb-4">
            <input
              type="text"
              value={customAmenity}
              onChange={(e) => setCustomAmenity(e.target.value)}
              placeholder="Add custom amenity..."
              className="flex-1 px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              onKeyPress={(e) => e.key === 'Enter' && addCustomAmenity()}
            />
            <button
              type="button"
              onClick={addCustomAmenity}
              className="px-4 py-2 bg-[#2B256D] text-white rounded-lg hover:bg-[#1f1a5a] transition-colors"
            >
              <Plus className="w-4 h-4" />
            </button>
          </div>
        </div>

        {/* Property Highlights */}
        <div>
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Property Highlights</h3>
          <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
            {propertyHighlights.map(highlight => (
              <label key={highlight} className="flex items-center space-x-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={highlights.includes(highlight)}
                  onChange={(e) => handleHighlightChange(highlight, e.target.checked)}
                  className="w-4 h-4 text-[#2B256D] border-gray-300 rounded focus:ring-[#2B256D]"
                />
                <span className="text-sm text-gray-700">{highlight}</span>
              </label>
            ))}
          </div>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Location & Media
        </button>
      </div>
    </motion.div>
  );
};