import React, { useState } from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { Upload, FileText, X, Shield, CheckCircle } from 'lucide-react';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { fileUtils } from '../../utils/fileUtils';
import { documentsApi } from '../../utils/documentsApi';

const documentTypes = [
  'Property Ownership Documents',
  'Building Approvals',
  'Property Tax Receipts',
  'NOC Certificates',
  'Floor Plans',
  'RERA Certificate',
  'Other'
];

export const DocumentsStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { setValue, watch, register } = useFormContext<ListingFormData>();
  const [selectedDocType, setSelectedDocType] = useState('');
  const [dragActive, setDragActive] = useState(false);
  const [reraVerifying, setReraVerifying] = useState(false);
  
  const documents = watch('documents') || [];
  const requiredDocs = watch('requiredDocs') || [];
  const reraNumber = watch('reraNumber') || '';
  const reraState = watch('reraState') || '';
  const reraVerified = watch('reraVerified') || false;

  const handleFileUpload = async (files: FileList | null, docType: string) => {
    if (!files || !docType) {
      console.log('📄 No files or docType provided');
      return;
    }

    console.log('📄 handleFileUpload called with:', {
      fileCount: files.length,
      docType,
      currentRequiredDocs: requiredDocs.length
    });

    const user = JSON.parse(localStorage.getItem('user') || '{}');
    const propertyId = watch('propertyId'); // Get actual property ID if editing
    
    for (const file of Array.from(files)) {
      console.log('📄 Processing file:', file.name, 'Size:', file.size);
      
      // Always add to form first for immediate UI feedback
      const newDoc = { file, type: docType, uploaded: false, uploading: true };
      const updatedRequiredDocs = [...requiredDocs, newDoc];
      const updatedDocuments = [...documents, file];
      
      console.log('📄 Adding to form - new requiredDocs length:', updatedRequiredDocs.length);
      
      setValue('requiredDocs', updatedRequiredDocs);
      setValue('documents', updatedDocuments);
      
      // If we have a property ID, upload immediately
      if (propertyId) {
        console.log('📄 Property ID exists, uploading immediately:', propertyId);
        try {
          const result = await documentsApi.uploadDocument(
            file, 
            propertyId, 
            user.user_id || user.id || 'anonymous', 
            docType.toLowerCase().replace(/\s+/g, '_')
          );
          
          // Update the document status
          const updatedDocs = updatedRequiredDocs.map(doc => 
            doc.file === file ? { ...doc, uploaded: true, uploading: false, documentId: result.id } : doc
          );
          setValue('requiredDocs', updatedDocs);
        } catch (error) {
          console.error('Upload failed:', error);
          // Update status to show upload failed
          const updatedDocs = updatedRequiredDocs.map(doc => 
            doc.file === file ? { ...doc, uploaded: false, uploading: false, error: 'Upload failed' } : doc
          );
          setValue('requiredDocs', updatedDocs);
        }
      } else {
        console.log('📄 No property ID yet - marking for upload after property creation');
        // No property ID yet - will upload after property creation
        const updatedDocs = updatedRequiredDocs.map(doc => 
          doc.file === file ? { ...doc, uploading: false, pendingUpload: true } : doc
        );
        setValue('requiredDocs', updatedDocs);
      }
    }
    
    console.log('📄 Final requiredDocs after upload:', watch('requiredDocs')?.length);
  };

  const removeDocument = (index: number) => {
    const newRequiredDocs = requiredDocs.filter((_, i) => i !== index);
    const newDocuments = documents.filter((_, i) => i !== index);
    
    setValue('requiredDocs', newRequiredDocs);
    setValue('documents', newDocuments);
  };

  const handleDrag = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (e.type === 'dragenter' || e.type === 'dragover') {
      setDragActive(true);
    } else if (e.type === 'dragleave') {
      setDragActive(false);
    }
  };

  const handleDrop = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setDragActive(false);
    
    if (e.dataTransfer.files && selectedDocType) {
      handleFileUpload(e.dataTransfer.files, selectedDocType);
    }
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Documents</h2>
        <p className="text-gray-600">Upload required property documents</p>
      </div>

      <div className="space-y-6">
        {/* RERA Verification Section */}
        <div className="bg-green-50 border border-green-200 rounded-lg p-6">
          <div className="flex items-center mb-4">
            <Shield className="w-6 h-6 text-green-600 mr-2" />
            <h3 className="text-lg font-medium text-green-900">RERA Verification</h3>
          </div>
          
          <div className="grid md:grid-cols-2 gap-4 mb-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                RERA Registration Number
              </label>
              <input
                {...register('reraNumber')}
                type="text"
                placeholder="e.g., P52100020240"
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent"
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                State
              </label>
              <select
                {...register('reraState')}
                className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent"
              >
                <option value="">Select State</option>
                <option value="Maharashtra">Maharashtra</option>
                <option value="Karnataka">Karnataka</option>
                <option value="Tamil Nadu">Tamil Nadu</option>
                <option value="Delhi">Delhi</option>
                <option value="Gujarat">Gujarat</option>
                <option value="Rajasthan">Rajasthan</option>
                <option value="Uttar Pradesh">Uttar Pradesh</option>
                <option value="West Bengal">West Bengal</option>
              </select>
            </div>
          </div>
          
          <button
            type="button"
            onClick={() => {
              if (reraNumber && reraState) {
                setReraVerifying(true);
                // Mock verification - in real app, call API
                setTimeout(() => {
                  setValue('reraVerified', true);
                  setReraVerifying(false);
                }, 2000);
              }
            }}
            disabled={!reraNumber || !reraState || reraVerifying}
            className="bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 disabled:opacity-50 disabled:cursor-not-allowed"
          >
            {reraVerifying ? 'Verifying...' : 'Verify RERA Number'}
          </button>
          
          {reraVerified && (
            <div className="flex items-center mt-3 text-green-700">
              <CheckCircle className="w-5 h-5 mr-2" />
              <span className="font-medium">RERA Number Verified</span>
            </div>
          )}
        </div>

        {/* Document Type Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Select Document Type
          </label>
          <select
            value={selectedDocType}
            onChange={(e) => setSelectedDocType(e.target.value)}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
          >
            <option value="">Choose document type...</option>
            {documentTypes.map(type => (
              <option key={type} value={type}>{type}</option>
            ))}
          </select>
        </div>

        {/* File Upload Area */}
        <div
          className={`border-2 border-dashed rounded-lg p-8 text-center transition-colors ${
            dragActive 
              ? 'border-[#2B256D] bg-blue-50' 
              : selectedDocType 
                ? 'border-gray-300 hover:border-[#2B256D] cursor-pointer' 
                : 'border-gray-200 cursor-not-allowed opacity-50'
          }`}
          onDragEnter={handleDrag}
          onDragLeave={handleDrag}
          onDragOver={handleDrag}
          onDrop={handleDrop}
        >
          <input
            type="file"
            multiple
            accept=".pdf"
            onChange={(e) => handleFileUpload(e.target.files, selectedDocType)}
            className="hidden"
            id="document-upload"
            disabled={!selectedDocType}
          />
          <label 
            htmlFor="document-upload" 
            className={selectedDocType ? 'cursor-pointer' : 'cursor-not-allowed'}
          >
            <Upload className="w-12 h-12 mx-auto mb-4 text-gray-400" />
            <div className="text-lg font-medium text-gray-900 mb-2">
              {selectedDocType ? `Upload ${selectedDocType}` : 'Select document type first'}
            </div>
            <div className="text-gray-600">
              Drag and drop files here, or click to browse
            </div>
            <div className="text-sm text-gray-500 mt-2">
              PDF files only, up to 10MB each
            </div>
          </label>
        </div>

        {/* Uploaded Documents */}
        {requiredDocs.length > 0 && (
          <div>
            <h3 className="text-lg font-medium text-gray-900 mb-4">
              Uploaded Documents ({requiredDocs.length})
            </h3>
            <div className="space-y-3">
              {requiredDocs.map((doc, index) => (
                <div key={index} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg border">
                  <div className="flex items-center">
                    <FileText className="w-8 h-8 text-[#2B256D] mr-3" />
                    <div>
                      <div className="font-medium text-gray-900">{doc.file.name}</div>
                      <div className="text-sm text-gray-600">
                        {doc.type} • {fileUtils.formatFileSize(doc.file.size)}
                        {doc.uploading && <span className="text-blue-600 ml-2">⏳ Uploading...</span>}
                        {doc.uploaded && <span className="text-green-600 ml-2">✓ Uploaded</span>}
                        {doc.error && <span className="text-red-600 ml-2">⚠ {doc.error}</span>}
                        {doc.pendingUpload && <span className="text-yellow-600 ml-2">📋 Will upload after property creation</span>}
                      </div>
                    </div>
                  </div>
                  <button
                    type="button"
                    onClick={() => removeDocument(index)}
                    className="text-red-600 hover:text-red-800 p-1"
                  >
                    <X className="w-5 h-5" />
                  </button>
                </div>
              ))}
            </div>
          </div>
        )}

        {/* Legal Compliance Notice */}
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <h4 className="font-medium text-yellow-900 mb-2">Legal Compliance Notice:</h4>
          <ul className="text-sm text-yellow-800 space-y-1">
            <li>• All documents are verified under RERA Act 2016</li>
            <li>• Property ownership documents are mandatory</li>
            <li>• Documents are stored securely as per IT Act 2000</li>
            <li>• Your data is protected under DPDP Act 2023</li>
          </ul>
        </div>

        {/* Document Requirements */}
        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <h4 className="font-medium text-blue-900 mb-2">Required Documents:</h4>
          <ul className="text-sm text-blue-800 space-y-1">
            <li>• Property ownership documents (mandatory)</li>
            <li>• RERA registration certificate</li>
            <li>• Building approvals and permits</li>
            <li>• Property tax receipts</li>
            <li>• NOC certificates (if applicable)</li>
            <li>• Floor plans or blueprints</li>
          </ul>
        </div>

        {/* Agreement & Notifications */}
        <div className="bg-red-50 border border-red-200 rounded-lg p-6">
          <h4 className="font-bold text-red-900 mb-4 text-lg">Agreement & Notifications</h4>
          
          <div className="space-y-4">
            <div className="flex items-start">
              <input
                {...register('termsAccepted')}
                type="checkbox"
                id="terms-conditions"
                className="mt-1 mr-3 h-4 w-4 text-red-600 focus:ring-red-500 border-gray-300 rounded"
                required
              />
              <label htmlFor="terms-conditions" className="text-sm text-gray-700">
                <span className="font-semibold text-red-800">I accept the Terms and Conditions *</span>
              </label>
            </div>
            
            <div className="ml-7 text-sm text-gray-700">
              <p className="font-medium mb-3">By checking this box, I confirm that —</p>
              
              <div className="space-y-3">
                <p>
                  <span className="font-medium">1.</span> I am the lawful owner or an authorized agent to post this property, and the information provided is true and accurate to the best of my knowledge. 
                  <span className="text-blue-600 font-medium">(Transfer of Property Act, 1882)</span>
                </p>
                
                <p>
                  <span className="font-medium">2.</span> I understand that the property listing must comply with all applicable real-estate regulations including the Real Estate (Regulation and Development) Act (RERA) 2016, and that non-compliance may lead to removal or penalties. 
                  <span className="text-blue-600 font-medium">(RERA Act 2016)</span>
                </p>
                
                <p>
                  <span className="font-medium">3.</span> I consent to the collection and processing of my personal and property data in accordance with the Digital Personal Data Protection (DPDP) Act 2023 and the Information Technology (IT) Act 2000. 
                  <span className="text-blue-600 font-medium">(Data Privacy Compliance)</span>
                </p>
                
                <p>
                  <span className="font-medium">4.</span> I acknowledge that NAL India is not liable for false listings or misrepresentations, and that any disputes arising will be governed under the Consumer Protection Act 2019.
                </p>
              </div>
            </div>
            
            <div className="flex items-start mt-4">
              <input
                {...register('nalTermsAccepted')}
                type="checkbox"
                id="nal-terms"
                className="mt-1 mr-3 h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300 rounded"
                required
              />
              <label htmlFor="nal-terms" className="text-sm text-gray-700">
                <span className="font-semibold text-green-800">☑️ I agree to NAL India's Terms of Service, Privacy Policy, and Property Listing Guidelines.</span>
              </label>
            </div>
          </div>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={onNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Location
        </button>
      </div>
    </motion.div>
  );
};