import React, { useCallback } from 'react';
import { useFormContext } from 'react-hook-form';

interface FixedInputFieldProps {
  name: string;
  label: string;
  type?: string;
  placeholder?: string;
  required?: boolean;
  validation?: any;
  className?: string;
}

export const FixedInputField: React.FC<FixedInputFieldProps> = ({
  name,
  label,
  type = 'text',
  placeholder,
  required = false,
  validation,
  className = ''
}) => {
  const { register, formState: { errors }, setValue, watch } = useFormContext();
  
  // Get current value without causing re-renders
  const currentValue = watch(name);
  
  // Stable onChange handler
  const handleChange = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
    setValue(name, e.target.value, { shouldValidate: true });
  }, [name, setValue]);

  const error = errors[name];

  return (
    <div>
      <label className="block text-sm font-medium text-gray-700 mb-2">
        {label} {required && <span className="required-asterisk">*</span>}
      </label>
      <input
        type={type}
        value={currentValue || ''}
        onChange={handleChange}
        className={`w-full px-3 py-2.5 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-[#0056D2] transition-colors bg-white ${className}`}
        placeholder={placeholder}
      />
      {error && <p className="mt-1 text-sm text-red-500">{error.message}</p>}
    </div>
  );
};