import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';
import { Home, Clock, Gavel, DollarSign } from 'lucide-react';

const listingTypes = [
  { value: 'sell', label: 'Sell', icon: Home, description: 'Sell your property' },
  { value: 'rent', label: 'Rent', icon: DollarSign, description: 'Rent out your property' },
  { value: 'urgent-sale', label: 'Urgent Sale', icon: Clock, description: 'Quick sale needed' },
  { value: 'bidding', label: 'Bidding', icon: Gavel, description: 'Auction your property' }
];

export const ListingTypeStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger, watch, setValue } = useFormContext<ListingFormData>();

  const selectedType = watch('listingType');

  const handleNext = async () => {
    const isValid = await trigger(['listingType']);
    if (isValid) onNext();
  };

  const handleTypeSelect = (type: string) => {
    setValue('listingType', type as any);
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Listing Type</h2>
        <p className="text-gray-600">How do you want to list your property?</p>
      </div>

      <div className="space-y-4">
        <input
          type="hidden"
          {...register('listingType', validation.required('Please select a listing type'))}
        />
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          {listingTypes.map((type) => {
            const Icon = type.icon;
            return (
              <button
                key={type.value}
                type="button"
                onClick={() => handleTypeSelect(type.value)}
                className={`p-6 border-2 rounded-lg text-left transition-all hover:shadow-md ${
                  selectedType === type.value
                    ? 'border-[#0056D2] bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className="flex items-start space-x-4">
                  <div className={`p-3 rounded-lg ${
                    selectedType === type.value ? 'bg-[#0056D2] text-white' : 'bg-gray-100 text-gray-600'
                  }`}>
                    <Icon className="w-6 h-6" />
                  </div>
                  <div>
                    <h3 className={`font-semibold ${
                      selectedType === type.value ? 'text-[#0056D2]' : 'text-gray-900'
                    }`}>
                      {type.label}
                    </h3>
                    <p className="text-sm text-gray-600 mt-1">{type.description}</p>
                  </div>
                </div>
              </button>
            );
          })}
        </div>

        {errors.listingType && (
          <p className="text-sm text-red-600">{errors.listingType.message}</p>
        )}
      </div>

      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Property Type
        </button>
      </div>
    </motion.div>
  );
};