import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';
import { AddressAutocomplete } from '../AddressAutocomplete';
import { LocationPicker } from '../LocationPicker';
import { MapPin } from 'lucide-react';

const cities = [
  'Mumbai', 'Delhi', 'Bangalore', 'Chennai', 'Kolkata', 'Hyderabad',
  'Pune', 'Ahmedabad', 'Surat', 'Jaipur', 'Lucknow', 'Kanpur'
];

export const LocationStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger, watch, setValue } = useFormContext<ListingFormData>();
  
  const address = watch('address') || '';
  const coordinates = watch('coordinates');
  const [isLocationPickerOpen, setIsLocationPickerOpen] = React.useState(false);

  const handleLocationSelect = (location: { lat: number; lng: number; address: string }) => {
    setValue('address', location.address);
    setValue('coordinates', { lat: location.lat, lng: location.lng });
  };

  const handleFindOnMap = () => {
    setIsLocationPickerOpen(true);
  };

  const handleNext = async () => {
    const isValid = await trigger(['address', 'city', 'pincode', 'locality', 'propertyAge', 'facing']);
    if (isValid) onNext();
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Location</h2>
        <p className="text-gray-600">Provide the complete address of your property</p>
      </div>

      <div className="space-y-6">
        {/* Address Autocomplete */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property Address *
          </label>
          <AddressAutocomplete
            value={address}
            onChange={(value) => setValue('address', value)}
            onLocationSelect={handleLocationSelect}
            placeholder="Start typing your property address..."
          />
          {errors.address && (
            <p className="mt-1 text-sm text-red-600">{errors.address.message}</p>
          )}
        </div>

        {/* City Selection */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            City *
          </label>
          <select
            {...register('city', validation.required('City is required'))}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
          >
            <option value="">Select City</option>
            {cities.map(city => (
              <option key={city} value={city}>{city}</option>
            ))}
          </select>
          {errors.city && (
            <p className="mt-1 text-sm text-red-600">{errors.city.message}</p>
          )}
        </div>

        {/* Find on Map Button */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Location on Map
          </label>
          <div className="space-y-3">
            <button
              type="button"
              onClick={handleFindOnMap}
              className="w-full flex items-center justify-center space-x-2 px-4 py-3 bg-[#2B256D] text-white rounded-lg hover:bg-[#1f1a5a] transition-colors"
            >
              <MapPin className="w-5 h-5" />
              <span className="font-medium">Find on Map</span>
            </button>
            
            {coordinates && (
              <div className="bg-green-50 border border-green-200 rounded-lg p-3">
                <p className="text-sm text-green-700 flex items-center">
                  <MapPin className="w-4 h-4 mr-2" />
                  Location selected: {coordinates.lat.toFixed(6)}, {coordinates.lng.toFixed(6)}
                </p>
                <button
                  type="button"
                  onClick={() => {
                    setValue('coordinates', undefined);
                  }}
                  className="mt-2 text-xs text-red-600 hover:text-red-800"
                >
                  Clear location
                </button>
              </div>
            )}
          </div>
        </div>

        {/* Additional Location Details */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Pincode *
            </label>
            <input
              type="text"
              {...register('pincode', validation.required('Pincode is required'))}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              placeholder="e.g., 560001"
            />
            {errors.pincode && (
              <p className="mt-1 text-sm text-red-600">{errors.pincode.message}</p>
            )}
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Locality *
            </label>
            <input
              type="text"
              {...register('locality', validation.required('Locality is required'))}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              placeholder="e.g., Koramangala"
            />
            {errors.locality && (
              <p className="mt-1 text-sm text-red-600">{errors.locality.message}</p>
            )}
          </div>
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Nearby Landmark
          </label>
          <input
            type="text"
            {...register('landmark')}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
            placeholder="e.g., Near Metro Station, Shopping Mall"
          />
        </div>

        {/* Property Details */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Property Age *
            </label>
            <select
              {...register('propertyAge', validation.required('Property age is required'))}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
            >
              <option value="">Select Age</option>
              <option value="Under Construction">Under Construction</option>
              <option value="0-1 years">0-1 years</option>
              <option value="1-5 years">1-5 years</option>
              <option value="5-10 years">5-10 years</option>
              <option value="10+ years">10+ years</option>
            </select>
            {errors.propertyAge && (
              <p className="mt-1 text-sm text-red-600">{errors.propertyAge.message}</p>
            )}
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Facing *
            </label>
            <select
              {...register('facing', validation.required('Facing direction is required'))}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
            >
              <option value="">Select Facing</option>
              <option value="North">North</option>
              <option value="South">South</option>
              <option value="East">East</option>
              <option value="West">West</option>
              <option value="North-East">North-East</option>
              <option value="North-West">North-West</option>
              <option value="South-East">South-East</option>
              <option value="South-West">South-West</option>
            </select>
            {errors.facing && (
              <p className="mt-1 text-sm text-red-600">{errors.facing.message}</p>
            )}
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Floor Number
            </label>
            <input
              type="text"
              {...register('floorNumber')}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              placeholder="e.g., 3rd Floor, Ground Floor"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Total Floors
            </label>
            <input
              type="text"
              {...register('totalFloors')}
              className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#2B256D] focus:border-transparent"
              placeholder="e.g., 10 floors"
            />
          </div>
        </div>
      </div>

      {/* Navigation */}
      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Documents
        </button>
      </div>
      
      {/* Location Picker Modal */}
      <LocationPicker
        isOpen={isLocationPickerOpen}
        onClose={() => setIsLocationPickerOpen(false)}
        onLocationSelect={handleLocationSelect}
        initialCoordinates={coordinates}
        initialAddress={address}
      />
    </motion.div>
  );
};