import React, { useState, useEffect } from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { Upload, Image, Video, X, Eye, Plus, Check } from 'lucide-react';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { mediaApi } from '../../utils/mediaApi';
import type { MediaGalleryItem } from '../../utils/mediaApi';
import { useAzureBlobMedia } from '../../hooks/useAzureBlobMedia';
import { toast } from 'sonner';
import { debugUserAuthentication } from '../../utils/debugAuth';

const imageCategoryOptions = [
  'exterior',
  'living_room',
  'kitchen',
  'bedroom',
  'bathroom',
  'balcony',
  'other'
];

export const MediaStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { setValue, watch } = useFormContext<ListingFormData>();
  const [activeTab, setActiveTab] = useState<'images' | 'videos'>('images');
  const [galleryItems, setGalleryItems] = useState<MediaGalleryItem[]>([]);
  const [selectedItems, setSelectedItems] = useState<number[]>([]);
  const [primaryImageId, setPrimaryImageId] = useState<number | null>(null);
  const [loading, setLoading] = useState(false);
  const [uploading, setUploading] = useState(false);
  const [previewImage, setPreviewImage] = useState<string | null>(null);
  const [useAzureBlob, setUseAzureBlob] = useState(true);
  
  const selectedMedia = watch('selectedMedia') || [];
  const primaryImage = watch('primaryImage') || null;
  
  // Get property ID for Azure Blob integration
  const propertyId = watch('id') || localStorage.getItem('current_property_id');
  
  // Azure Blob media hook
  const {
    uploadImages,
    uploadVideos
  } = useAzureBlobMedia();

  useEffect(() => {
    loadGallery();
  }, [activeTab]);

  const loadGallery = async () => {
    try {
      setLoading(true);
      // For new properties, start with empty gallery since we store files temporarily
      setGalleryItems([]);
    } catch (error) {
      console.error('Gallery load error:', error);
      setGalleryItems([]);
    } finally {
      setLoading(false);
    }
  };

  const handleFileUpload = async (files: FileList | null) => {
    if (!files) return;

    // Debug user authentication before upload
    console.log('🔍 MediaStep: Starting file upload, checking user authentication...');
    debugUserAuthentication();

    try {
      setUploading(true);
      const fileArray = Array.from(files);
      let uploadedItems = [];
      
      // Store files in memory for upload after property creation
      const currentFiles = Array.isArray((window as any).pendingMediaFiles) 
        ? (window as any).pendingMediaFiles 
        : [];
      
      if (activeTab === 'images') {
        const imageFiles = fileArray.filter(file => file.type.startsWith('image/'));
        
        // Check current image count limit (5 images max)
        const currentImageCount = selectedMedia.filter(item => item.media_type === 'image').length;
        const availableSlots = 5 - currentImageCount;
        
        if (availableSlots <= 0) {
          toast.error('Maximum 5 images allowed per property');
          setUploading(false);
          return;
        }
        
        if (imageFiles.length > availableSlots) {
          toast.error(`You can only upload ${availableSlots} more image(s). Maximum 5 images allowed.`);
          setUploading(false);
          return;
        }
        
        for (let i = 0; i < imageFiles.length; i++) {
          const file = imageFiles[i];
          const tempId = Date.now() + i;
          const tempUrl = URL.createObjectURL(file);
          
          // Store file in memory instead of localStorage
          currentFiles.push({
            tempId,
            file: file,
            type: 'image',
            fileName: file.name,
            fileType: file.type,
            fileSize: file.size
          });
          
          // Store files directly for PostgreSQL upload
          uploadedItems.push({
            id: tempId,
            media_type: 'image',
            file: file,  // Store actual File object
            alt_text: file.name,
            category: 'exterior',
            isPending: true
          });
        }
      } else {
        const videoFiles = fileArray.filter(file => file.type.startsWith('video/'));
        
        for (let i = 0; i < videoFiles.length; i++) {
          const file = videoFiles[i];
          const tempId = Date.now() + i;
          const tempUrl = URL.createObjectURL(file);
          
          // Store file in memory instead of localStorage
          currentFiles.push({
            tempId,
            file: file,
            type: 'video',
            fileName: file.name,
            fileType: file.type,
            fileSize: file.size
          });
          
          uploadedItems.push({
            id: tempId,
            media_type: 'video',
            file: file,  // Store actual File object
            alt_text: file.name,
            isPending: true
          });
        }
      }
      
      // Store in window object instead of localStorage
      (window as any).pendingMediaFiles = currentFiles;
      
      if (uploadedItems.length > 0) {
        // Add to gallery items
        setGalleryItems(prev => [...uploadedItems, ...prev]);
        
        // Auto-select uploaded items
        const newItemIds = uploadedItems.map(item => item.id);
        setSelectedItems(prev => [...prev, ...newItemIds]);
        
        // Update form data
        const currentSelected = selectedMedia || [];
        const newSelectedMedia = [...currentSelected, ...uploadedItems];
        setValue('selectedMedia', newSelectedMedia);
        
        // Set first uploaded image as primary if no primary exists
        if (activeTab === 'images' && !primaryImageId && uploadedItems.length > 0) {
          setPrimaryImageId(uploadedItems[0].id);
          setValue('primaryImage', uploadedItems[0]);
        }
        
        toast.success(`${uploadedItems.length} file(s) ready for upload`);
      }
    } catch (error) {
      console.error('Upload error:', error);
      toast.error('Upload failed');
    } finally {
      setUploading(false);
    }
  };

  const toggleItemSelection = (itemId: number) => {
    setSelectedItems(prev => {
      const newSelection = prev.includes(itemId)
        ? prev.filter(id => id !== itemId)
        : [...prev, itemId];
      
      // Update form data
      const selectedGalleryItems = galleryItems.filter(item => newSelection.includes(item.id));
      setValue('selectedMedia', selectedGalleryItems);
      
      return newSelection;
    });
  };

  const setPrimaryImage = (itemId: number) => {
    setPrimaryImageId(itemId);
    const primaryItem = galleryItems.find(item => item.id === itemId);
    setValue('primaryImage', primaryItem || null);
    
    // Auto-select if not already selected
    if (!selectedItems.includes(itemId)) {
      toggleItemSelection(itemId);
    }
  };

  const formatCategoryName = (category: string) => {
    return category.split('_').map(word => 
      word.charAt(0).toUpperCase() + word.slice(1)
    ).join(' ');
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Media</h2>
        <p className="text-gray-600">Upload images and videos to showcase your property</p>
      </div>

      <div className="space-y-6">
        {/* Media Type Tabs */}
        <div className="flex space-x-1 bg-gray-100 p-1 rounded-lg">
          <button
            type="button"
            onClick={() => setActiveTab('images')}
            className={`flex-1 py-2 px-4 rounded-md text-sm font-medium transition-colors ${
              activeTab === 'images'
                ? 'bg-white text-[#2B256D] shadow-sm'
                : 'text-gray-600 hover:text-gray-900'
            }`}
          >
            <Image className="w-4 h-4 inline mr-2" />
            Images
          </button>
          <button
            type="button"
            onClick={() => setActiveTab('videos')}
            className={`flex-1 py-2 px-4 rounded-md text-sm font-medium transition-colors ${
              activeTab === 'videos'
                ? 'bg-white text-[#2B256D] shadow-sm'
                : 'text-gray-600 hover:text-gray-900'
            }`}
          >
            <Video className="w-4 h-4 inline mr-2" />
            Videos
          </button>
        </div>

        {/* Upload Section */}
        <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center hover:border-[#2B256D] transition-colors">
          {activeTab === 'images' && selectedMedia.filter(item => item.media_type === 'image').length >= 5 ? (
            <div className="text-center">
              <Image className="w-12 h-12 mx-auto mb-4 text-gray-400" />
              <div className="text-lg font-medium text-gray-900 mb-2">
                Maximum Images Reached
              </div>
              <div className="text-gray-600">
                You have uploaded the maximum of 5 images
              </div>
              <div className="text-sm text-red-500 mt-2">
                Remove an image to upload a new one
              </div>
            </div>
          ) : (
            <>
              <input
                type="file"
                multiple
                accept={activeTab === 'images' ? 'image/*' : 'video/*'}
                onChange={(e) => handleFileUpload(e.target.files)}
                className="hidden"
                id="media-upload"
                disabled={uploading}
              />
              <label htmlFor="media-upload" className="cursor-pointer">
                {uploading ? (
                  <div className="flex items-center justify-center">
                    <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#2B256D]"></div>
                    <span className="ml-2">Uploading...</span>
                  </div>
                ) : (
                  <>
                    {activeTab === 'images' ? (
                      <Image className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                    ) : (
                      <Video className="w-12 h-12 mx-auto mb-4 text-gray-400" />
                    )}
                    <div className="text-lg font-medium text-gray-900 mb-2">
                      Upload {activeTab === 'images' ? 'Images' : 'Videos'} to Gallery
                    </div>
                    <div className="text-gray-600">
                      Add new {activeTab} to your media gallery
                    </div>
                    <div className="text-sm text-gray-500 mt-2">
                      {activeTab === 'images' 
                        ? `JPG, PNG, WebP up to 5MB each (${5 - selectedMedia.filter(item => item.media_type === 'image').length} remaining)`
                        : 'MP4, MOV, AVI up to 50MB each'
                      }
                    </div>
                  </>
                )}
              </label>
            </>
          )}
        </div>



        {/* Selected Media Display */}
        {selectedMedia && selectedMedia.filter(item => item.media_type === (activeTab === 'images' ? 'image' : 'video')).length > 0 && (
          <div>
            <div className="flex items-center justify-between mb-4">
              <h3 className="text-lg font-semibold text-gray-900">
                Selected {activeTab === 'images' ? 'Images' : 'Videos'} ({selectedMedia.filter(item => item.media_type === (activeTab === 'images' ? 'image' : 'video')).length}{activeTab === 'images' ? '/5' : ''})
              </h3>
              <button
                type="button"
                onClick={() => {
                  const filteredMedia = selectedMedia.filter(item => item.media_type !== (activeTab === 'images' ? 'image' : 'video'));
                  const filteredIds = filteredMedia.map(item => item.id);
                  setSelectedItems(filteredIds);
                  setValue('selectedMedia', filteredMedia);
                  if (activeTab === 'images') {
                    setPrimaryImageId(null);
                    setValue('primaryImage', null);
                  }
                }}
                className="text-sm text-red-600 hover:text-red-800"
              >
                Clear All {activeTab === 'images' ? 'Images' : 'Videos'}
              </button>
            </div>
            
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-5 gap-4 mb-4">
              {selectedMedia.filter(item => item.media_type === (activeTab === 'images' ? 'image' : 'video')).map((item) => {
                if (!item) return null;
                const isSelected = selectedItems.includes(item.id);
                const isPrimary = primaryImageId === item.id;
                
                
                return (
                  <div key={item.id} className="relative group">
                    <div className={`relative rounded-lg overflow-hidden border-2 ${
                      isPrimary ? 'border-yellow-500 ring-2 ring-yellow-200' : 'border-gray-200'
                    }`}>
                      {item.media_type === 'image' ? (
                        <img
                          src={item.file && item.file instanceof File ? URL.createObjectURL(item.file) : '/placeholder-image.jpg'}
                          alt={item.alt_text || 'Selected item'}
                          className="w-full h-24 object-cover"
                        />
                      ) : (
                        <video
                          src={item.file && item.file instanceof File ? URL.createObjectURL(item.file) : ''}
                          className="w-full h-24 object-cover"
                          muted
                          preload="metadata"
                        />
                      )}
                      
                      {/* Primary badge */}
                      {isPrimary && (
                        <div className="absolute top-1 left-1 bg-yellow-500 text-white text-xs px-2 py-1 rounded">
                          Primary
                        </div>
                      )}
                      
                      {/* Remove button */}
                      <button
                        type="button"
                        onClick={() => {
                          const newSelection = selectedItems.filter(id => id !== item.id);
                          setSelectedItems(newSelection);
                          
                          if (primaryImageId === item.id) {
                            setPrimaryImageId(null);
                            setValue('primaryImage', null);
                          }
                          
                          const selectedGalleryItems = galleryItems.filter(gi => newSelection.includes(gi.id));
                          setValue('selectedMedia', selectedGalleryItems);
                        }}
                        className="absolute top-1 right-1 w-6 h-6 bg-red-500 text-white rounded-full flex items-center justify-center opacity-0 group-hover:opacity-100 transition-opacity hover:bg-red-600"
                      >
                        <X className="w-4 h-4" />
                      </button>
                      
                      {/* Set as primary button (images only) */}
                      {activeTab === 'images' && !isPrimary && (
                        <button
                          type="button"
                          onClick={() => setPrimaryImage(item.id)}
                          className="absolute bottom-1 left-1 right-1 bg-black bg-opacity-75 text-white text-xs py-1 px-2 rounded opacity-0 group-hover:opacity-100 transition-opacity"
                        >
                          Set Primary
                        </button>
                      )}
                    </div>
                    
                    {/* Category label */}
                    {item.category && (
                      <p className="text-xs text-gray-500 mt-1 text-center">
                        {formatCategoryName(item.category)}
                      </p>
                    )}
                  </div>
                );
              })}
            </div>
            
            {/* Summary info */}
            <div className="bg-blue-50 border border-blue-200 rounded-lg p-3">
              <div className="flex items-center justify-between text-sm">
                <span className="text-blue-800">
                  {selectedMedia.filter(item => item.media_type === (activeTab === 'images' ? 'image' : 'video')).length}{activeTab === 'images' ? '/5' : ''} {activeTab} selected for this property
                </span>
                {activeTab === 'images' && primaryImageId && (
                  <span className="text-yellow-700 font-medium">
                    ✓ Primary image set
                  </span>
                )}
              </div>
            </div>
          </div>
        )}

        {/* Media Tips */}
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <h4 className="font-medium text-yellow-900 mb-2">Media Tips:</h4>
          <ul className="text-sm text-yellow-800 space-y-1">
            <li>• Upload high-quality, well-lit images</li>
            <li>• Include photos of all rooms and exterior</li>
            <li>• Videos should be stable and well-focused</li>
            <li>• First image will be used as the main property photo</li>
            <li>• Maximum 5 images allowed per property</li>
          </ul>
        </div>
      </div>

      {/* Image Preview Modal */}
      {previewImage && (
        <div className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50">
          <div className="relative max-w-4xl max-h-full p-4">
            <img src={previewImage} alt="Preview" className="max-w-full max-h-full object-contain" />
            <button
              onClick={() => setPreviewImage(null)}
              className="absolute top-4 right-4 text-white hover:text-gray-300"
            >
              <X className="w-8 h-8" />
            </button>
          </div>
        </div>
      )}


    </motion.div>
  );
};