import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';

export const PersonalDetailsStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger } = useFormContext<ListingFormData>();

  const handleNext = async () => {
    const isValid = await trigger(['ownerName', 'mobile', 'email']);
    if (isValid) onNext();
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Personal Details</h2>
        <p className="text-gray-600">Tell us about yourself</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Full Name *
          </label>
          <input
            type="text"
            {...register('ownerName', validation.required('Name is required'))}
            className="w-full px-6 py-4 h-14 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
            placeholder="Enter your full name"
          />
          {errors.ownerName && (
            <p className="mt-1 text-sm text-red-600">{errors.ownerName.message}</p>
          )}
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Company Name (Optional)
          </label>
          <input
            type="text"
            {...register('companyName')}
            className="w-full px-6 py-4 h-14 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
            placeholder="Enter company name if any"
          />
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Mobile Number *
          </label>
          <input
            type="tel"
            {...register('mobile', {
              ...validation.required('Mobile number is required'),
              pattern: {
                value: /^[6-9]\d{9}$/,
                message: 'Please enter a valid mobile number'
              }
            })}
            className="w-full px-6 py-4 h-14 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
            placeholder="Enter your mobile number"
          />
          {errors.mobile && (
            <p className="mt-1 text-sm text-red-600">{errors.mobile.message}</p>
          )}
        </div>

        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Email Address *
          </label>
          <input
            type="email"
            {...register('email', {
              ...validation.required('Email is required'),
              pattern: {
                value: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
                message: 'Please enter a valid email address'
              }
            })}
            className="w-full px-6 py-4 h-14 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
            placeholder="Enter your email address"
          />
          {errors.email && (
            <p className="mt-1 text-sm text-red-600">{errors.email.message}</p>
          )}
        </div>
      </div>

      <div className="flex justify-between pt-6">
        <div>
          {!isFirst && (
            <button
              type="button"
              onClick={onPrev}
              className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
            >
              Previous
            </button>
          )}
        </div>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Property Title
        </button>
      </div>
    </motion.div>
  );
};