import React from 'react';
import { useFormContext } from 'react-hook-form';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { Button } from '../ui/button';
import { Input } from '../ui/input';
import { Label } from '../ui/label';
import { X, Plus } from 'lucide-react';

const PREDEFINED_TAGS = [
  'Swimming Pool', 'Gym', 'Parking', 'Security', 'Garden', 'Balcony',
  'Elevator', 'Power Backup', 'Water Supply', 'Gated Community',
  'Club House', 'Children Play Area', 'Jogging Track', 'CCTV',
  'Intercom', 'Maintenance Staff', 'Visitor Parking', 'Fire Safety'
];

export function PropertyTagsStep() {
  const { watch, setValue } = useFormContext();
  const selectedTags = watch('propertyTags') || [];
  const [customTag, setCustomTag] = React.useState('');

  const toggleTag = (tag: string) => {
    const current = selectedTags || [];
    const updated = current.includes(tag)
      ? current.filter((t: string) => t !== tag)
      : [...current, tag];
    setValue('propertyTags', updated);
  };

  const addCustomTag = () => {
    if (customTag.trim() && !selectedTags.includes(customTag.trim())) {
      setValue('propertyTags', [...selectedTags, customTag.trim()]);
      setCustomTag('');
    }
  };

  const removeTag = (tag: string) => {
    setValue('propertyTags', selectedTags.filter((t: string) => t !== tag));
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle>Property Tags & Features</CardTitle>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Predefined Tags */}
          <div>
            <Label className="text-base font-medium mb-3 block">
              Select Property Features
            </Label>
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-2">
              {PREDEFINED_TAGS.map((tag) => (
                <Button
                  key={tag}
                  type="button"
                  variant={selectedTags.includes(tag) ? "default" : "outline"}
                  size="sm"
                  onClick={() => toggleTag(tag)}
                  className="justify-start text-sm"
                >
                  {tag}
                </Button>
              ))}
            </div>
          </div>

          {/* Custom Tag Input */}
          <div>
            <Label className="text-base font-medium mb-3 block">
              Add Custom Features
            </Label>
            <div className="flex gap-2">
              <Input
                placeholder="Enter custom feature..."
                value={customTag}
                onChange={(e) => setCustomTag(e.target.value)}
                onKeyPress={(e) => e.key === 'Enter' && (e.preventDefault(), addCustomTag())}
              />
              <Button
                type="button"
                onClick={addCustomTag}
                disabled={!customTag.trim()}
                size="sm"
              >
                <Plus className="w-4 h-4" />
              </Button>
            </div>
          </div>

          {/* Selected Tags Display */}
          {selectedTags.length > 0 && (
            <div>
              <Label className="text-base font-medium mb-3 block">
                Selected Features ({selectedTags.length})
              </Label>
              <div className="flex flex-wrap gap-2">
                {selectedTags.map((tag: string) => (
                  <Badge
                    key={tag}
                    variant="secondary"
                    className="flex items-center gap-1 px-3 py-1"
                  >
                    {tag}
                    <Button
                      type="button"
                      variant="ghost"
                      size="sm"
                      onClick={() => removeTag(tag)}
                      className="h-auto p-0 ml-1 hover:bg-transparent"
                    >
                      <X className="w-3 h-3" />
                    </Button>
                  </Badge>
                ))}
              </div>
            </div>
          )}
        </CardContent>
      </Card>
    </div>
  );
}