import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';

export const PropertyTitleStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger, watch } = useFormContext<ListingFormData>();

  const handleNext = async () => {
    const isValid = await trigger(['title']);
    if (isValid) onNext();
  };

  const title = watch('title');

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Title</h2>
        <p className="text-gray-600">Create an attractive title for your property</p>
      </div>

      <div className="space-y-6">
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Property Title *
          </label>
          <input
            type="text"
            {...register('title', {
              ...validation.required('Property title is required'),
              ...validation.minLength(10, 'Title must be at least 10 characters'),
              ...validation.maxLength(100, 'Title must not exceed 100 characters')
            })}
            className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-[#0056D2] focus:border-transparent"
            placeholder="e.g., Luxury 3BHK Apartment with Sea View in Prime Location"
          />
          {errors.title && (
            <p className="mt-1 text-sm text-red-600">{errors.title.message}</p>
          )}
          <div className="mt-2 flex justify-between text-sm text-gray-500">
            <span>Make it descriptive and attractive</span>
            <span>{title?.length || 0}/100</span>
          </div>
        </div>

        <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
          <h3 className="font-medium text-blue-900 mb-2">Tips for a great title:</h3>
          <ul className="text-sm text-blue-800 space-y-1">
            <li>• Include property type (e.g., 2BHK, Villa, Office)</li>
            <li>• Mention key features (e.g., Sea View, Garden, Parking)</li>
            <li>• Add location highlights (e.g., Prime Location, Near Metro)</li>
            <li>• Keep it under 100 characters</li>
          </ul>
        </div>
      </div>

      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Listing Type
        </button>
      </div>
    </motion.div>
  );
};