import React from 'react';
import { useFormContext } from 'react-hook-form';
import { motion } from 'framer-motion';
import { ListingFormData, FormStepProps } from '../../types/listing';
import { validation } from '../../utils/validation';
import { Building, Home, TreePine, Building2, Users } from 'lucide-react';

const propertyTypes = [
  { value: 'apartment', label: 'Apartment', icon: Building, description: 'Flat in a building' },
  { value: 'villa', label: 'Villa', icon: Home, description: 'Independent villa' },
  { value: 'house', label: 'House', icon: TreePine, description: 'Independent house' },
  { value: 'commercial', label: 'Commercial', icon: Building2, description: 'Office, shop, warehouse' },
  { value: 'pg-coliving', label: 'PG/Co-living', icon: Users, description: 'Paying guest accommodation' }
];

export const PropertyTypeStep: React.FC<FormStepProps> = ({ onNext, onPrev, isFirst, isLast }) => {
  const { register, formState: { errors }, trigger, watch, setValue } = useFormContext<ListingFormData>();

  const selectedType = watch('propertyType');

  const handleNext = async () => {
    const isValid = await trigger(['propertyType']);
    if (isValid) onNext();
  };

  const handleTypeSelect = (type: string) => {
    setValue('propertyType', type as any);
  };

  return (
    <motion.div
      initial={{ opacity: 0, x: 20 }}
      animate={{ opacity: 1, x: 0 }}
      exit={{ opacity: 0, x: -20 }}
      className="space-y-6"
    >
      <div>
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Property Type</h2>
        <p className="text-gray-600">What type of property are you listing?</p>
      </div>

      <div className="space-y-4">
        <input
          type="hidden"
          {...register('propertyType', validation.required('Please select a property type'))}
        />
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {propertyTypes.map((type) => {
            const Icon = type.icon;
            return (
              <button
                key={type.value}
                type="button"
                onClick={() => handleTypeSelect(type.value)}
                className={`p-6 border-2 rounded-lg text-center transition-all hover:shadow-md ${
                  selectedType === type.value
                    ? 'border-[#0056D2] bg-blue-50'
                    : 'border-gray-200 hover:border-gray-300'
                }`}
              >
                <div className={`w-12 h-12 mx-auto mb-3 p-3 rounded-lg ${
                  selectedType === type.value ? 'bg-[#0056D2] text-white' : 'bg-gray-100 text-gray-600'
                }`}>
                  <Icon className="w-6 h-6" />
                </div>
                <h3 className={`font-semibold mb-1 ${
                  selectedType === type.value ? 'text-[#0056D2]' : 'text-gray-900'
                }`}>
                  {type.label}
                </h3>
                <p className="text-sm text-gray-600">{type.description}</p>
              </button>
            );
          })}
        </div>

        {errors.propertyType && (
          <p className="text-sm text-red-600">{errors.propertyType.message}</p>
        )}
      </div>

      <div className="flex justify-between pt-6">
        <button
          type="button"
          onClick={onPrev}
          className="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors font-medium"
        >
          Previous
        </button>
        <button
          type="button"
          onClick={handleNext}
          className="px-6 py-3 bg-[#0056D2] text-white rounded-lg hover:bg-[#0056D2]/90 transition-colors font-medium"
        >
          Next: Property Details
        </button>
      </div>
    </motion.div>
  );
};