import React, { useState, useEffect } from 'react';
import { MapPin, Calendar, TrendingUp } from 'lucide-react';
import { futureInsightsService, InfraProject } from '../../api/futureInsights';

interface BuyerInfraMapProps {
  city?: string;
  locality?: string;
}

export function BuyerInfraMap({ city, locality }: BuyerInfraMapProps) {
  const [projects, setProjects] = useState<InfraProject[]>([]);
  const [loading, setLoading] = useState(false);
  const [selectedProject, setSelectedProject] = useState<InfraProject | null>(null);

  useEffect(() => {
    loadProjects();
  }, [city, locality]);

  const loadProjects = async () => {
    setLoading(true);
    try {
      console.log('Loading ML-powered projects for:', { city, locality });
      
      // Call real API with ML predictions
      const response = await futureInsightsService.fetchInfraProjects({
        city: city || undefined,
        locality: locality || undefined
      });
      
      console.log('ML Projects loaded:', response);
      setProjects(response);
    } catch (error) {
      console.error('Error loading ML projects:', error);
      setProjects([]);
    } finally {
      setLoading(false);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'completed': return 'bg-green-100 text-green-800';
      case 'in_progress': return 'bg-blue-100 text-blue-800';
      case 'approved': return 'bg-yellow-100 text-yellow-800';
      case 'planned': return 'bg-gray-100 text-gray-800';
      case 'delayed': return 'bg-red-100 text-red-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  if (loading) {
    return (
      <div className="p-6 bg-white rounded-lg shadow">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <MapPin className="w-5 h-5 mr-2 text-blue-600" />
          Infrastructure Projects
        </h3>
        <div className="animate-pulse">
          <div className="space-y-3">
            {[1, 2, 3].map(i => (
              <div key={i} className="h-20 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white p-8">


      <div className="space-y-4">
        {projects.map((project) => (
          <div
            key={project.id}
            className="border border-gray-100 rounded-md p-4 hover:bg-gray-50 transition-colors cursor-pointer mb-3 last:mb-0"
            onClick={() => setSelectedProject(project)}
          >
            <div className="flex justify-between items-start mb-2">
              <h4 className="font-medium text-gray-900">{project.project_name}</h4>
              <span className={`px-2 py-1 rounded text-xs font-medium ${getStatusColor(project.status)}`}>
                {project.status.replace('_', ' ')}
              </span>
            </div>
            
            <div className="flex items-center text-sm text-gray-500 mb-3">
              <span className="capitalize">{project.project_type.replace('_', ' ')}</span>
              {project.locality && (
                <>
                  <span className="mx-2">•</span>
                  <span>{project.locality}</span>
                </>
              )}
            </div>

            <div className="flex items-center justify-between text-sm">
              <div className="flex items-center text-gray-500">
                <Calendar className="w-4 h-4 mr-1" />
                {project.expected_completion_date ? 
                  new Date(project.expected_completion_date).toLocaleDateString() : 
                  'TBD'
                }
              </div>
              
              {project.impact_score && (
                <div className="flex items-center text-green-600">
                  <TrendingUp className="w-4 h-4 mr-1" />
                  <span className="font-medium">
                    {project.impact_score}/10
                  </span>
                </div>
              )}
            </div>

            {project.construction_progress > 0 && (
              <div className="mt-3">
                <div className="flex justify-between text-xs text-gray-600 mb-1">
                  <span>Progress</span>
                  <span>{project.construction_progress}%</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div
                    className="bg-blue-600 h-2 rounded-full"
                    style={{ width: `${project.construction_progress}%` }}
                  ></div>
                </div>
              </div>
            )}
          </div>
        ))}

        {projects.length === 0 && !loading && (
          <div className="text-center py-8 text-gray-500">
            <MapPin className="w-12 h-12 mx-auto mb-4 text-gray-300" />
            <p>No infrastructure projects found for this area</p>
            <p className="text-xs mt-2">Try selecting Mumbai or Bangalore from filters</p>
          </div>
        )}
      </div>

      {selectedProject && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg p-6 max-w-md w-full mx-4">
            <h3 className="text-lg font-semibold mb-4">{selectedProject.project_name}</h3>
            <div className="space-y-2 text-sm">
              <p><span className="font-medium">Type:</span> {selectedProject.project_type.replace('_', ' ')}</p>
              <p><span className="font-medium">Status:</span> {selectedProject.status.replace('_', ' ')}</p>
              <p><span className="font-medium">Location:</span> {selectedProject.locality}, {selectedProject.city}</p>
              {selectedProject.expected_completion_date && (
                <p><span className="font-medium">Expected Completion:</span> {new Date(selectedProject.expected_completion_date).toLocaleDateString()}</p>
              )}
              {selectedProject.impact_score && (
                <p><span className="font-medium">Impact Score:</span> {selectedProject.impact_score}/10</p>
              )}
            </div>
            <button
              onClick={() => setSelectedProject(null)}
              className="mt-4 w-full bg-blue-600 text-white py-2 rounded hover:bg-blue-700"
            >
              Close
            </button>
          </div>
        </div>
      )}
    </div>
  );
}