import React, { useState, useEffect } from 'react';
import { TrendingUp, BarChart3, Info } from 'lucide-react';
import { futureInsightsService, PriceForecast } from '../../api/futureInsights';

interface ForecastChartProps {
  city?: string;
  locality?: string;
  propertyType?: string;
}

export function ForecastChart({ city, locality, propertyType }: ForecastChartProps) {
  const [forecasts, setForecasts] = useState<PriceForecast[]>([]);
  const [loading, setLoading] = useState(false);
  const [selectedPeriod, setSelectedPeriod] = useState('1_year');

  const periods = [
    { value: '6_months', label: '6 Months' },
    { value: '1_year', label: '1 Year' },
    { value: '2_years', label: '2 Years' },
    { value: '5_years', label: '5 Years' }
  ];

  useEffect(() => {
    loadForecasts();
  }, [city, locality, propertyType, selectedPeriod]);

  const loadForecasts = async () => {
    setLoading(true);
    try {
      console.log('Loading ML forecasts for:', { city, locality, propertyType, selectedPeriod });
      
      // Call real ML API for price forecasts
      const response = await futureInsightsService.fetchPriceForecasts({
        city: city || undefined,
        locality: locality || undefined,
        property_type: propertyType || undefined,
        forecast_period: selectedPeriod
      });
      
      console.log('ML Forecasts loaded:', response);
      setForecasts(response);
    } catch (error) {
      console.error('Error loading ML forecasts:', error);
      setForecasts([]);
    } finally {
      setLoading(false);
    }
  };

  const formatPrice = (price: number) => {
    if (price >= 10000000) {
      return `₹${(price / 10000000).toFixed(1)}Cr`;
    } else if (price >= 100000) {
      return `₹${(price / 100000).toFixed(1)}L`;
    }
    return `₹${price.toLocaleString()}`;
  };

  const getConfidenceColor = (score: number) => {
    if (score >= 0.8) return 'text-green-600';
    if (score >= 0.6) return 'text-yellow-600';
    return 'text-red-600';
  };

  if (loading) {
    return (
      <div className="p-6 bg-white rounded-lg shadow">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <BarChart3 className="w-5 h-5 mr-2 text-blue-600" />
          Price Forecasts
        </h3>
        <div className="animate-pulse">
          <div className="h-32 bg-gray-200 rounded"></div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white p-8">
      <div className="flex justify-end mb-6">
        <select
          value={selectedPeriod}
          onChange={(e) => setSelectedPeriod(e.target.value)}
          className="border-2 border-gray-200 rounded-xl px-4 py-2 text-sm bg-gray-50 focus:outline-none focus:border-green-500 font-medium"
        >
        {periods.map(period => (
          <option key={period.value} value={period.value}>
            {period.label}
          </option>
        ))}
      </select>
      </div>
      
      <div className="space-y-4">
        {forecasts.map((forecast, index) => (
          <div key={index} className="border border-gray-100 rounded-md p-4">
            <div className="flex justify-between items-start mb-3">
              <div>
                <h4 className="font-medium text-gray-900">{forecast.locality}</h4>
                <p className="text-sm text-gray-600 capitalize">
                  {forecast.property_type.replace('_', ' ')}
                </p>
              </div>
              
              {forecast.confidence_score && (
                <div className="flex items-center text-sm">
                  <Info className="w-4 h-4 mr-1" />
                  <span className={`font-medium ${getConfidenceColor(forecast.confidence_score)}`}>
                    {Math.round(forecast.confidence_score * 100)}% confidence
                  </span>
                </div>
              )}
            </div>

            <div className="grid grid-cols-2 gap-6 mb-4">
              <div>
                <div>
                  <p className="text-xs text-gray-500 mb-1">Current Price</p>
                  <p className="text-lg font-semibold text-gray-900">
                  {forecast.current_avg_price ? formatPrice(forecast.current_avg_price) : 'N/A'}
                </p>
              </div>
              
              <div>
                </div>
              </div>
              
              <div>
                <div>
                  <p className="text-xs text-gray-500 mb-1">Forecasted Price</p>
                  <p className="text-lg font-semibold text-green-600">
                  {forecast.forecasted_price ? formatPrice(forecast.forecasted_price) : 'N/A'}
                  </p>
                </div>
              </div>
            </div>

            {forecast.appreciation_percentage && (
              <div className="flex items-center justify-between">
                <div className="flex items-center">
                  <TrendingUp className="w-4 h-4 mr-1 text-green-600" />
                  <span className="text-sm font-medium text-green-600">
                    +{forecast.appreciation_percentage.toFixed(1)}% growth
                  </span>
                </div>
                
                {forecast.infra_impact_score && (
                  <div className="text-sm text-gray-600">
                    Infra Impact: {forecast.infra_impact_score}/10
                  </div>
                )}
              </div>
            )}

            {forecast.forecast_factors && (
              <div className="mt-3 pt-3 border-t">
                <p className="text-xs text-gray-500 mb-2">Key Factors:</p>
                <div className="flex flex-wrap gap-2">
                  {forecast.forecast_factors.infra_projects?.map((project: string, i: number) => (
                    <span key={i} className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded">
                      {project}
                    </span>
                  ))}
                </div>
              </div>
            )}
          </div>
        ))}

        {forecasts.length === 0 && !loading && (
          <div className="text-center py-8 text-gray-500">
            <BarChart3 className="w-12 h-12 mx-auto mb-4 text-gray-300" />
            <p>No forecast data available for the selected criteria</p>
            <p className="text-xs mt-2">Try selecting Mumbai or Bangalore from filters</p>
          </div>
        )}
      </div>
    </div>
  );
}