import React, { useState, useEffect } from 'react';
import { Flame, TrendingUp, MapPin, Star } from 'lucide-react';
import { futureInsightsService, Hotspot } from '../../api/futureInsights';

interface HotspotPanelProps {
  city?: string;
  userRole?: string;
}

export function HotspotPanel({ city, userRole }: HotspotPanelProps) {
  const [hotspots, setHotspots] = useState<Hotspot[]>([]);
  const [loading, setLoading] = useState(false);
  const [filter, setFilter] = useState('all');

  useEffect(() => {
    loadHotspots();
  }, [city, userRole, filter]);

  const loadHotspots = async () => {
    setLoading(true);
    try {
      console.log('Loading AI hotspots for:', { city, userRole, filter });
      
      // Call real ML API for hotspots
      const response = await futureInsightsService.fetchHotspots({
        city: city || undefined,
        growth_potential: filter !== 'all' ? filter : undefined,
        user_role: userRole || undefined
      });
      
      console.log('AI Hotspots loaded:', response);
      setHotspots(response);
    } catch (error) {
      console.error('Error loading AI hotspots:', error);
      setHotspots([]);
    } finally {
      setLoading(false);
    }
  };

  const getScoreColor = (score: number) => {
    if (score >= 80) return 'text-green-600 bg-green-100';
    if (score >= 60) return 'text-yellow-600 bg-yellow-100';
    return 'text-red-600 bg-red-100';
  };

  const getPotentialIcon = (potential: string) => {
    switch (potential) {
      case 'high': return <Flame className="w-4 h-4 text-red-500" />;
      case 'medium': return <TrendingUp className="w-4 h-4 text-yellow-500" />;
      case 'low': return <MapPin className="w-4 h-4 text-gray-500" />;
      default: return <MapPin className="w-4 h-4 text-gray-500" />;
    }
  };

  if (loading) {
    return (
      <div className="p-6 bg-white rounded-lg shadow">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Flame className="w-5 h-5 mr-2 text-red-600" />
          Development Hotspots
        </h3>
        <div className="animate-pulse">
          <div className="space-y-3">
            {[1, 2, 3].map(i => (
              <div key={i} className="h-24 bg-gray-200 rounded"></div>
            ))}
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white p-8">
      <div className="flex justify-end mb-6">
        <select
          value={filter}
          onChange={(e) => setFilter(e.target.value)}
          className="border-2 border-gray-200 rounded-xl px-4 py-2 text-sm bg-gray-50 focus:outline-none focus:border-purple-500 font-medium"
        >
        <option value="all">All Potential</option>
        <option value="high">High Potential</option>
        <option value="medium">Medium Potential</option>
        <option value="low">Low Potential</option>
      </select>
      </div>
      
      <div className="space-y-4">
        {hotspots.map((hotspot, index) => (
          <div key={index} className="border border-gray-100 rounded-md p-4 hover:bg-gray-50 transition-colors">
            <div className="flex justify-between items-start mb-3">
              <div>
                <h4 className="font-medium text-gray-900 flex items-center">
                  {getPotentialIcon(hotspot.growth_potential)}
                  <span className="ml-2">{hotspot.locality}</span>
                </h4>
                <p className="text-sm text-gray-600">{hotspot.city}</p>
              </div>
              
              {hotspot.hotspot_score && (
                <div className={`px-3 py-1 rounded text-sm font-semibold ${getScoreColor(hotspot.hotspot_score)}`}>
                  {Math.round(hotspot.hotspot_score)}/100
                </div>
              )}
            </div>

            <div className="grid grid-cols-2 gap-4 mb-4 text-sm text-gray-600">
              <div>
                <span>Growth Potential:</span>
                <span className="ml-2 font-medium text-gray-900 capitalize">{hotspot.growth_potential}</span>
              </div>
              
              <div>
                <span>Upcoming Projects:</span>
                <span className="ml-2 font-medium text-gray-900">{hotspot.upcoming_projects_count}</span>
              </div>
              
              <div>
                <span>Demand Trend:</span>
                <span className="ml-2 font-medium text-gray-900 capitalize">{hotspot.demand_trend}</span>
              </div>
              
              <div>
                <span>Investment Ready:</span>
                <span className="ml-2 font-medium text-gray-900 capitalize">{hotspot.investment_readiness}</span>
              </div>
            </div>

            {hotspot.key_drivers && hotspot.key_drivers.length > 0 && (
              <div className="mb-3">
                <p className="text-xs text-gray-500 mb-2">Key Drivers:</p>
                <div className="flex flex-wrap gap-2">
                  {hotspot.key_drivers.map((driver, i) => (
                    <span key={i} className="px-2 py-1 bg-green-100 text-green-800 text-xs rounded">
                      {driver}
                    </span>
                  ))}
                </div>
              </div>
            )}

            {hotspot.risk_factors && hotspot.risk_factors.length > 0 && (
              <div className="mb-3">
                <p className="text-xs text-gray-500 mb-2">Risk Factors:</p>
                <div className="flex flex-wrap gap-2">
                  {hotspot.risk_factors.map((risk, i) => (
                    <span key={i} className="px-2 py-1 bg-red-100 text-red-800 text-xs rounded">
                      {risk}
                    </span>
                  ))}
                </div>
              </div>
            )}

            {hotspot.recommended_for && hotspot.recommended_for.length > 0 && (
              <div className="flex items-center justify-between">
                <div className="flex items-center text-sm text-gray-600">
                  <Star className="w-4 h-4 mr-1" />
                  <span>Recommended for: {hotspot.recommended_for.join(', ')}</span>
                </div>
                
                {hotspot.infra_density_score && (
                  <div className="text-sm text-gray-600">
                    Infra Density: {hotspot.infra_density_score}/10
                  </div>
                )}
              </div>
            )}
          </div>
        ))}

        {hotspots.length === 0 && !loading && (
          <div className="text-center py-8 text-gray-500">
            <Flame className="w-12 h-12 mx-auto mb-4 text-gray-300" />
            <p>No hotspots found for the selected criteria</p>
            <p className="text-xs mt-2">Try selecting Mumbai or Bangalore from filters</p>
          </div>
        )}
      </div>
    </div>
  );
}