import React from 'react';
import { FloodRiskData } from '../../types/gee';
import { getFloodBgColor } from '../../utils/floodRiskCalculator';
import { Droplets, Mountain, MapPin, AlertTriangle, Info } from 'lucide-react';

interface FloodRiskCardProps {
  floodData: FloodRiskData;
  propertyTitle?: string;
}

export function FloodRiskCard({ floodData, propertyTitle }: FloodRiskCardProps) {
  const riskPercentage = (floodData.score / 10) * 100;

  return (
    <div className="bg-white rounded-lg shadow-lg border border-gray-200 p-6 space-y-4">
      {propertyTitle && (
        <h3 className="text-lg font-semibold text-gray-900">{propertyTitle}</h3>
      )}

      <div className="flex items-center justify-between">
        <div className="flex items-center gap-2">
          <Droplets className="w-5 h-5 text-blue-600" />
          <span className="text-sm font-medium text-gray-700">Flood Risk Assessment</span>
        </div>
        <span className={`px-3 py-1 rounded-full text-xs font-semibold border ${getFloodBgColor(floodData.score)}`}>
          {floodData.level.toUpperCase()}
        </span>
      </div>

      <div className="space-y-2">
        <div className="flex items-center justify-between">
          <span className="text-2xl font-bold text-gray-900">{floodData.score}/10</span>
          <span className="text-sm text-gray-500">Risk Score</span>
        </div>
        
        <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
          <div
            className={`h-full transition-all duration-500 ${
              floodData.score <= 3 ? 'bg-green-500' :
              floodData.score <= 6 ? 'bg-yellow-500' : 'bg-red-500'
            }`}
            style={{ width: `${riskPercentage}%` }}
          />
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4 pt-4 border-t">
        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <Mountain className="w-4 h-4" />
            <span className="text-xs">Elevation</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{floodData.elevation}m</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <MapPin className="w-4 h-4" />
            <span className="text-xs">Water Distance</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{(floodData.waterDistance / 1000).toFixed(1)}km</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <AlertTriangle className="w-4 h-4" />
            <span className="text-xs">Historical Floods</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{floodData.historicalFloods} events</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <Droplets className="w-4 h-4" />
            <span className="text-xs">Water Occurrence</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{floodData.waterOccurrence}%</p>
        </div>
      </div>

      <div className="bg-blue-50 border border-blue-200 rounded-lg p-3 mt-4">
        <div className="flex items-start gap-2">
          <Info className="w-4 h-4 text-blue-600 mt-0.5 flex-shrink-0" />
          <div className="text-xs text-blue-800">
            <p className="font-medium mb-1">What does this mean?</p>
            <p className="text-blue-700">
              {floodData.score <= 3 && 'This property has low flood risk. The area is at higher elevation with minimal historical flooding.'}
              {floodData.score > 3 && floodData.score <= 6 && 'This property has moderate flood risk. Consider flood insurance and check drainage systems.'}
              {floodData.score > 6 && 'This property has high flood risk. Flood insurance is strongly recommended. Verify flood protection measures.'}
            </p>
          </div>
        </div>
      </div>

      <div className="text-xs text-gray-500 pt-2 border-t">
        Data source: Google Earth Engine (Satellite imagery analysis)
        <br />
        Last updated: {new Date(floodData.lastUpdated).toLocaleDateString()}
      </div>
    </div>
  );
}

export default FloodRiskCard;
