import React from 'react';
import { FloodRiskData } from '../../types/gee';
import { getFloodColor } from '../../utils/floodRiskCalculator';

interface FloodRiskLayerProps {
  properties: Array<{
    id: number | string;
    latitude: number;
    longitude: number;
    floodData?: FloodRiskData;
  }>;
  visible: boolean;
  onPropertyClick?: (propertyId: number | string) => void;
}

export function FloodRiskLayer({ properties, visible, onPropertyClick }: FloodRiskLayerProps) {
  if (!visible) return null;

  return (
    <>
      {properties.map((property) => {
        if (!property.floodData) return null;

        const color = getFloodColor(property.floodData.score);
        const opacity = 0.3 + (property.floodData.score / 10) * 0.4;

        return (
          <div
            key={property.id}
            className="absolute transform -translate-x-1/2 -translate-y-1/2 cursor-pointer group"
            style={{
              left: `${((property.longitude - 72) / 20) * 100}%`,
              top: `${((28 - property.latitude) / 20) * 100}%`,
              zIndex: 10
            }}
            onClick={() => onPropertyClick?.(property.id)}
          >
            <div
              className="rounded-full transition-all duration-300 group-hover:scale-125"
              style={{
                width: '40px',
                height: '40px',
                backgroundColor: color,
                opacity: opacity,
                border: `2px solid ${color}`,
                boxShadow: `0 0 20px ${color}`
              }}
            />
            
            <div className="absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 opacity-0 group-hover:opacity-100 transition-opacity pointer-events-none">
              <div className="bg-black text-white text-xs rounded px-2 py-1 whitespace-nowrap">
                <div className="font-semibold">Flood Risk: {property.floodData.score}/10</div>
                <div className="text-gray-300">{property.floodData.level.toUpperCase()}</div>
              </div>
              <div className="absolute top-full left-1/2 transform -translate-x-1/2">
                <div className="w-0 h-0 border-l-4 border-r-4 border-t-4 border-l-transparent border-r-transparent border-t-black"></div>
              </div>
            </div>
          </div>
        );
      })}
    </>
  );
}

export default FloodRiskLayer;
