import React from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '../ui/card';
import { Badge } from '../ui/badge';
import { InfrastructureData } from '../../types/gee';
import { getConnectivityColor, formatDistance } from '../../utils/infrastructureCalculator';
import { 
  MapPin, 
  Train, 
  Plane, 
  Building2, 
  Cross, 
  GraduationCap,
  Navigation,
  TrendingUp,
  Home,
  Briefcase
} from 'lucide-react';

interface InfrastructureCardProps {
  infrastructureData: InfrastructureData;
  propertyTitle: string;
}

export const InfrastructureCard: React.FC<InfrastructureCardProps> = ({
  infrastructureData,
  propertyTitle
}) => {
  const getConnectivityBadge = () => {
    const colors = {
      excellent: 'bg-green-100 text-green-800 border-green-300',
      good: 'bg-blue-100 text-blue-800 border-blue-300',
      average: 'bg-orange-100 text-orange-800 border-orange-300',
      poor: 'bg-red-100 text-red-800 border-red-300'
    };

    const labels = {
      excellent: '🌟 Excellent',
      good: '✅ Good',
      average: '⚠️ Average',
      poor: '❌ Poor'
    };

    return (
      <Badge className={`${colors[infrastructureData.connectivityLevel]} border`}>
        {labels[infrastructureData.connectivityLevel]}
      </Badge>
    );
  };

  const getScoreBar = (score: number, label: string, icon: React.ReactNode) => {
    const percentage = (score / 10) * 100;
    const color = score >= 8 ? 'bg-green-500' : score >= 6 ? 'bg-blue-500' : score >= 4 ? 'bg-orange-500' : 'bg-red-500';

    return (
      <div className="space-y-2">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            {icon}
            <span className="text-sm font-medium text-gray-700">{label}</span>
          </div>
          <span className="text-sm font-bold text-gray-900">{score.toFixed(1)}/10</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div 
            className={`${color} h-2 rounded-full transition-all duration-300`}
            style={{ width: `${percentage}%` }}
          />
        </div>
      </div>
    );
  };

  return (
    <Card className="shadow-lg border-2 border-indigo-200 bg-white">
      <CardHeader className="pb-3 bg-gradient-to-r from-indigo-50 to-purple-50">
        <div className="flex items-start justify-between">
          <div className="flex-1">
            <CardTitle className="text-lg font-bold text-gray-900 mb-1">
              Infrastructure & Accessibility
            </CardTitle>
            <p className="text-sm text-gray-600 line-clamp-1">{propertyTitle}</p>
          </div>
          {getConnectivityBadge()}
        </div>
      </CardHeader>

      <CardContent className="space-y-4 pt-4">
        {/* Overall Accessibility Score */}
        <div className="bg-indigo-50 rounded-lg p-4 border border-indigo-200">
          <div className="flex items-center justify-between mb-2">
            <div className="flex items-center gap-2">
              <TrendingUp className="w-5 h-5 text-indigo-600" />
              <span className="font-semibold text-gray-900">Accessibility Score</span>
            </div>
            <span className="text-2xl font-bold text-indigo-600">
              {infrastructureData.accessibilityScore.toFixed(1)}/10
            </span>
          </div>
          <div className="w-full bg-indigo-200 rounded-full h-3">
            <div 
              className="bg-indigo-600 h-3 rounded-full transition-all duration-300"
              style={{ width: `${(infrastructureData.accessibilityScore / 10) * 100}%` }}
            />
          </div>
        </div>

        {/* Category Scores */}
        <div className="space-y-3">
          {getScoreBar(
            infrastructureData.transportScore, 
            'Transport', 
            <Navigation className="w-4 h-4 text-blue-600" />
          )}
          {getScoreBar(
            infrastructureData.amenitiesScore, 
            'Amenities', 
            <Home className="w-4 h-4 text-green-600" />
          )}
          {getScoreBar(
            infrastructureData.employmentScore, 
            'Employment', 
            <Briefcase className="w-4 h-4 text-purple-600" />
          )}
        </div>

        {/* Detailed Distances */}
        <div className="border-t pt-4 space-y-3">
          <h4 className="font-semibold text-sm text-gray-900 mb-3">Nearby Infrastructure</h4>
          
          {/* Highway */}
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center gap-2">
              <MapPin className="w-4 h-4 text-gray-500" />
              <span className="text-gray-700">{infrastructureData.nearestHighway.name}</span>
            </div>
            <span className="font-medium text-gray-900">
              {formatDistance(infrastructureData.nearestHighway.distance)}
            </span>
          </div>

          {/* Metro */}
          {infrastructureData.nearestMetro && (
            <div className="flex items-center justify-between text-sm">
              <div className="flex items-center gap-2">
                <Train className="w-4 h-4 text-blue-600" />
                <span className="text-gray-700">{infrastructureData.nearestMetro.name}</span>
              </div>
              <span className="font-medium text-gray-900">
                {formatDistance(infrastructureData.nearestMetro.distance)}
              </span>
            </div>
          )}

          {/* Airport */}
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center gap-2">
              <Plane className="w-4 h-4 text-indigo-600" />
              <span className="text-gray-700">{infrastructureData.nearestAirport.name}</span>
            </div>
            <span className="font-medium text-gray-900">
              {formatDistance(infrastructureData.nearestAirport.distance)}
            </span>
          </div>

          {/* IT Park */}
          {infrastructureData.nearestITPark && (
            <div className="flex items-center justify-between text-sm">
              <div className="flex items-center gap-2">
                <Building2 className="w-4 h-4 text-purple-600" />
                <span className="text-gray-700">{infrastructureData.nearestITPark.name}</span>
              </div>
              <span className="font-medium text-gray-900">
                {formatDistance(infrastructureData.nearestITPark.distance)}
              </span>
            </div>
          )}

          {/* Hospital */}
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center gap-2">
              <Cross className="w-4 h-4 text-red-600" />
              <span className="text-gray-700">{infrastructureData.nearestHospital.name}</span>
            </div>
            <span className="font-medium text-gray-900">
              {formatDistance(infrastructureData.nearestHospital.distance)}
            </span>
          </div>

          {/* School */}
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center gap-2">
              <GraduationCap className="w-4 h-4 text-green-600" />
              <span className="text-gray-700">{infrastructureData.nearestSchool.name}</span>
            </div>
            <span className="font-medium text-gray-900">
              {formatDistance(infrastructureData.nearestSchool.distance)}
            </span>
          </div>
        </div>

        {/* Explanation */}
        <div className="bg-gray-50 rounded-lg p-3 border border-gray-200">
          <p className="text-xs text-gray-600">
            {infrastructureData.connectivityLevel === 'excellent' && 
              '🌟 Exceptional connectivity! This location offers premium access to all major infrastructure and amenities.'}
            {infrastructureData.connectivityLevel === 'good' && 
              '✅ Well-connected location with good access to essential infrastructure and amenities.'}
            {infrastructureData.connectivityLevel === 'average' && 
              '⚠️ Moderate connectivity. Some infrastructure may require additional travel time.'}
            {infrastructureData.connectivityLevel === 'poor' && 
              '❌ Limited connectivity. Consider transportation needs carefully.'}
          </p>
        </div>

        {/* Data Source */}
        <div className="text-xs text-gray-500 text-center pt-2 border-t">
          Data from spatial analysis • Updated {new Date(infrastructureData.lastUpdated).toLocaleDateString()}
        </div>
      </CardContent>
    </Card>
  );
};
