import React from 'react';
import { UrbanGrowthData } from '../../types/gee';
import { getGrowthBgColor } from '../../utils/urbanGrowthCalculator';
import { TrendingUp, Building2, BarChart3, Target, Info } from 'lucide-react';

interface UrbanGrowthCardProps {
  growthData: UrbanGrowthData;
  propertyTitle?: string;
}

export function UrbanGrowthCard({ growthData, propertyTitle }: UrbanGrowthCardProps) {
  const growthPercentage = (growthData.growthScore / 10) * 100;

  return (
    <div className="bg-white rounded-lg shadow-lg border border-gray-200 p-6 space-y-4">
      {propertyTitle && (
        <h3 className="text-lg font-semibold text-gray-900">{propertyTitle}</h3>
      )}

      <div className="flex items-center justify-between">
        <div className="flex items-center gap-2">
          <TrendingUp className="w-5 h-5 text-green-600" />
          <span className="text-sm font-medium text-gray-700">Urban Growth Analysis</span>
        </div>
        <span className={`px-3 py-1 rounded-full text-xs font-semibold border ${getGrowthBgColor(growthData.developmentLevel)}`}>
          {growthData.developmentLevel.toUpperCase()}
        </span>
      </div>

      <div className="space-y-2">
        <div className="flex items-center justify-between">
          <span className="text-2xl font-bold text-gray-900">{growthData.growthScore.toFixed(1)}/10</span>
          <span className="text-sm text-gray-500">Growth Score</span>
        </div>
        
        <div className="w-full bg-gray-200 rounded-full h-3 overflow-hidden">
          <div
            className="h-full transition-all duration-500 bg-gradient-to-r from-green-400 to-green-600"
            style={{ width: `${growthPercentage}%` }}
          />
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4 pt-4 border-t">
        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <BarChart3 className="w-4 h-4" />
            <span className="text-xs">Growth Rate</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{growthData.growthRate.toFixed(1)}%</p>
          <p className="text-xs text-gray-500">per year</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <Building2 className="w-4 h-4" />
            <span className="text-xs">Urban Expansion</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{growthData.urbanAreaChange.toFixed(1)}%</p>
          <p className="text-xs text-gray-500">since 2015</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <Target className="w-4 h-4" />
            <span className="text-xs">Infrastructure</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{growthData.infrastructureScore}/10</p>
        </div>

        <div className="space-y-1">
          <div className="flex items-center gap-2 text-gray-600">
            <TrendingUp className="w-4 h-4" />
            <span className="text-xs">Investment Potential</span>
          </div>
          <p className="text-lg font-semibold text-gray-900">{growthData.investmentPotential.toFixed(1)}/10</p>
        </div>
      </div>

      <div className="bg-green-50 border border-green-200 rounded-lg p-3 mt-4">
        <div className="flex items-start gap-2">
          <Info className="w-4 h-4 text-green-600 mt-0.5 flex-shrink-0" />
          <div className="text-xs text-green-800">
            <p className="font-medium mb-1">What does this mean?</p>
            <p className="text-green-700">
              {growthData.developmentLevel === 'emerging' && 'This area is experiencing rapid growth with high investment potential. Ideal for long-term appreciation.'}
              {growthData.developmentLevel === 'developing' && 'This area is steadily developing with good infrastructure. Balanced risk and return.'}
              {growthData.developmentLevel === 'mature' && 'This area is well-developed with stable growth. Lower risk, steady returns.'}
              {growthData.developmentLevel === 'saturated' && 'This area is fully developed with limited growth potential. Focus on location premium.'}
            </p>
          </div>
        </div>
      </div>

      <div className="text-xs text-gray-500 pt-2 border-t">
        Data source: Satellite imagery analysis (2015-2024)
        <br />
        Last updated: {new Date(growthData.lastUpdated).toLocaleDateString()}
      </div>
    </div>
  );
}

export default UrbanGrowthCard;
