import React, { useEffect, useRef, useState } from "react";
import L from "leaflet";
import "leaflet/dist/leaflet.css";

interface Facility {
  name?: string;
  category?: string;
  latitude?: number;
  longitude?: number;
  distance_km?: number;
}

interface FacilityGroup {
  category: string;
  count: number;
  items: Facility[];
}

interface FacilitiesMapProps {
  centerLat: number;
  centerLon: number;
  facilities: FacilityGroup[];
  radiusKm: number;
}

const categoryColors: Record<string, string> = {
  schools: "#3B82F6",
  healthcare: "#EF4444",
  grocery: "#10B981",
  transit: "#F59E0B",
  parks: "#22C55E",
  banks: "#8B5CF6",
  restaurants: "#EC4899",
  police: "#1E40AF",
  worship: "#7C3AED",
  pharmacies: "#DC2626",
  atms: "#9333EA",
  malls: "#DB2777",
  fitness: "#059669",
  cinema: "#EA580C",
  library: "#0891B2",
  fuel: "#CA8A04",
};

const categoryIcons: Record<string, string> = {
  schools: "🏫",
  healthcare: "🏥",
  grocery: "🛒",
  transit: "🚇",
  parks: "🌳",
  banks: "🏦",
  restaurants: "🍽️",
  police: "👮",
  worship: "🕌",
  pharmacies: "💊",
  atms: "🏧",
  malls: "🏬",
  fitness: "💪",
  cinema: "🎬",
  library: "📚",
  fuel: "⛽",
};

export function FacilitiesMap({
  centerLat,
  centerLon,
  facilities,
  radiusKm,
}: FacilitiesMapProps) {
  const mapRef = useRef<HTMLDivElement>(null);
  const mapInstanceRef = useRef<L.Map | null>(null);
  const markersRef = useRef<L.Marker[]>([]);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);

  useEffect(() => {
    if (!mapRef.current || mapInstanceRef.current) return;

    // Initialize map
    const map = L.map(mapRef.current).setView([centerLat, centerLon], 13);
    mapInstanceRef.current = map;

    // Add OpenStreetMap tiles
    L.tileLayer("https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png", {
      attribution: "© OpenStreetMap contributors",
      maxZoom: 19,
    }).addTo(map);

    // Add center marker (search location)
    const centerIcon = L.divIcon({
      html: '<div style="background: #0056D2; width: 20px; height: 20px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>',
      className: "",
      iconSize: [20, 20],
      iconAnchor: [10, 10],
    });
    L.marker([centerLat, centerLon], { icon: centerIcon })
      .addTo(map)
      .bindPopup("<b>Search Location</b>");

    // Add radius circle
    L.circle([centerLat, centerLon], {
      radius: radiusKm * 1000,
      color: "#0056D2",
      fillColor: "#0056D2",
      fillOpacity: 0.1,
      weight: 2,
    }).addTo(map);

    return () => {
      map.remove();
      mapInstanceRef.current = null;
    };
  }, [centerLat, centerLon, radiusKm]);

  useEffect(() => {
    if (!mapInstanceRef.current) return;

    const map = mapInstanceRef.current;

    // Clear existing markers
    markersRef.current.forEach((marker) => marker.remove());
    markersRef.current = [];

    // Filter facilities based on selected category
    const filteredFacilities = selectedCategory
      ? facilities.filter((group) => group.category === selectedCategory)
      : facilities;

    // Add facility markers
    filteredFacilities.forEach((group) => {
      const color = categoryColors[group.category] || "#6B7280";
      const emoji = categoryIcons[group.category] || "📍";

      group.items.forEach((facility) => {
        if (facility.latitude && facility.longitude) {
          const icon = L.divIcon({
            html: `<div style="font-size: 24px; text-shadow: 0 2px 4px rgba(0,0,0,0.3);">${emoji}</div>`,
            className: "",
            iconSize: [24, 24],
            iconAnchor: [12, 12],
          });

          const marker = L.marker([facility.latitude, facility.longitude], {
            icon,
          }).addTo(map).bindPopup(`
              <div style="min-width: 150px;">
                <b>${facility.name}</b><br/>
                <span style="color: ${color}; font-weight: 600;">${group.category}</span><br/>
                <span style="color: #6B7280; font-size: 12px;">${facility.distance_km?.toFixed(2)} km away</span>
              </div>
            `);

          markersRef.current.push(marker);
        }
      });
    });
  }, [facilities, selectedCategory]);

  return (
    <div className="space-y-4">
      {/* Category Filter Buttons */}
      <div style={{ display: 'flex', flexWrap: 'wrap', gap: '8px' }}>
        <button
          onClick={() => setSelectedCategory(null)}
          style={{
            display: 'flex',
            alignItems: 'center',
            gap: '8px',
            padding: '8px 16px',
            borderRadius: '8px',
            fontWeight: '500',
            fontSize: '14px',
            transition: 'all 0.2s',
            border: 'none',
            cursor: 'pointer',
            backgroundColor: selectedCategory === null ? '#2563eb' : '#f3f4f6',
            color: selectedCategory === null ? '#ffffff' : '#374151',
            boxShadow: selectedCategory === null ? '0 4px 6px -1px rgba(0, 0, 0, 0.1)' : 'none'
          }}
          onMouseEnter={(e) => {
            if (selectedCategory !== null) {
              e.currentTarget.style.backgroundColor = '#e5e7eb';
            }
          }}
          onMouseLeave={(e) => {
            if (selectedCategory !== null) {
              e.currentTarget.style.backgroundColor = '#f3f4f6';
            }
          }}
        >
          <span>All Facilities</span>
        </button>
        {facilities.map((group) => (
          <button
            key={group.category}
            onClick={() => setSelectedCategory(group.category)}
            style={{
              display: 'flex',
              alignItems: 'center',
              gap: '8px',
              padding: '8px 16px',
              borderRadius: '8px',
              fontWeight: '500',
              fontSize: '14px',
              transition: 'all 0.2s',
              border: 'none',
              cursor: 'pointer',
              backgroundColor: selectedCategory === group.category ? '#2563eb' : '#f3f4f6',
              color: selectedCategory === group.category ? '#ffffff' : '#374151',
              boxShadow: selectedCategory === group.category ? '0 4px 6px -1px rgba(0, 0, 0, 0.1)' : 'none'
            }}
            onMouseEnter={(e) => {
              if (selectedCategory !== group.category) {
                e.currentTarget.style.backgroundColor = '#e5e7eb';
              }
            }}
            onMouseLeave={(e) => {
              if (selectedCategory !== group.category) {
                e.currentTarget.style.backgroundColor = '#f3f4f6';
              }
            }}
          >
            <span style={{ fontSize: '18px' }}>
              {categoryIcons[group.category] || "📍"}
            </span>
            <span style={{ textTransform: 'capitalize' }}>{group.category}</span>
            <span style={{ fontSize: '12px', opacity: 0.9 }}>({group.count})</span>
          </button>
        ))}
      </div>

      {/* Map */}
      <div
        ref={mapRef}
        style={{
          height: "500px",
          width: "100%",
          borderRadius: "12px",
          overflow: "hidden",
        }}
      />
    </div>
  );
}
