import React, { useState } from 'react';
import { Plus, X, BarChart3, TrendingUp, MapPin } from 'lucide-react';
import { Property } from './types';

interface ComparisonArea {
  id: string;
  name: string;
  bounds: { north: number; south: number; east: number; west: number };
  properties: Property[];
  color: string;
}

interface AreaComparisonProps {
  properties: Property[];
  onAreaSelect: (callback: (bounds: any) => void) => void;
}

const COMPARISON_COLORS = [
  '#3B82F6', '#EF4444', '#10B981', '#F59E0B', 
  '#8B5CF6', '#EC4899', '#06B6D4', '#84CC16'
];

export function AreaComparison({ properties, onAreaSelect }: AreaComparisonProps) {
  const [comparisonAreas, setComparisonAreas] = useState<ComparisonArea[]>([]);
  const [isAddingArea, setIsAddingArea] = useState(false);

  const addComparisonArea = () => {
    if (comparisonAreas.length >= 4) return; // Max 4 areas

    setIsAddingArea(true);
    onAreaSelect((bounds) => {
      const newArea: ComparisonArea = {
        id: `area-${Date.now()}`,
        name: `Area ${comparisonAreas.length + 1}`,
        bounds,
        properties: properties.filter(p => 
          p.latitude >= bounds.south && p.latitude <= bounds.north &&
          p.longitude >= bounds.west && p.longitude <= bounds.east
        ),
        color: COMPARISON_COLORS[comparisonAreas.length % COMPARISON_COLORS.length]
      };
      
      setComparisonAreas(prev => [...prev, newArea]);
      setIsAddingArea(false);
    });
  };

  const removeArea = (id: string) => {
    setComparisonAreas(prev => prev.filter(area => area.id !== id));
  };

  const updateAreaName = (id: string, name: string) => {
    setComparisonAreas(prev => 
      prev.map(area => area.id === id ? { ...area, name } : area)
    );
  };

  const calculateMetrics = (areaProperties: Property[]) => {
    if (areaProperties.length === 0) {
      return {
        totalProperties: 0,
        averagePrice: 0,
        medianPrice: 0,
        pricePerSqft: 0,
        density: 0
      };
    }

    const prices = areaProperties
      .map(p => p.sale_price || (p.monthly_rent * 12) || 0)
      .filter(price => price > 0)
      .sort((a, b) => a - b);

    const averagePrice = prices.reduce((a, b) => a + b, 0) / prices.length;
    const medianPrice = prices[Math.floor(prices.length / 2)] || 0;

    const pricePerSqftValues = areaProperties
      .filter(p => (p.built_up_area || p.carpet_area || p.super_area) && (p.sale_price || p.monthly_rent))
      .map(p => {
        const price = p.sale_price || (p.monthly_rent * 12);
        const area = p.built_up_area || p.carpet_area || p.super_area || 1;
        return price / area;
      });

    const avgPricePerSqft = pricePerSqftValues.length > 0 
      ? pricePerSqftValues.reduce((a, b) => a + b, 0) / pricePerSqftValues.length 
      : 0;

    return {
      totalProperties: areaProperties.length,
      averagePrice,
      medianPrice,
      pricePerSqft: avgPricePerSqft,
      density: areaProperties.length // Simplified density calculation
    };
  };

  const formatPrice = (price: number): string => {
    if (price >= 10000000) return `₹${(price / 10000000).toFixed(1)}Cr`;
    if (price >= 100000) return `₹${(price / 100000).toFixed(1)}L`;
    return `₹${price.toLocaleString()}`;
  };

  const getWinner = (metric: string): string | null => {
    if (comparisonAreas.length < 2) return null;

    const values = comparisonAreas.map(area => {
      const metrics = calculateMetrics(area.properties);
      switch (metric) {
        case 'price': return metrics.averagePrice;
        case 'density': return metrics.density;
        case 'pricePerSqft': return metrics.pricePerSqft;
        default: return 0;
      }
    });

    const maxIndex = values.indexOf(Math.max(...values));
    return comparisonAreas[maxIndex]?.id || null;
  };

  if (comparisonAreas.length === 0) {
    return (
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
        <div className="flex items-center justify-between mb-4">
          <h3 className="text-lg font-semibold text-gray-900">Area Comparison</h3>
          <button
            onClick={addComparisonArea}
            disabled={isAddingArea}
            className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50"
          >
            <Plus className="w-4 h-4" />
            Add Area
          </button>
        </div>
        
        <div className="text-center py-8 text-gray-500">
          <BarChart3 className="w-12 h-12 mx-auto mb-3 text-gray-300" />
          <p className="mb-2">Compare different areas side by side</p>
          <p className="text-sm">Select areas on the map to analyze market differences</p>
        </div>
      </div>
    );
  }

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900">Area Comparison</h3>
        <button
          onClick={addComparisonArea}
          disabled={isAddingArea || comparisonAreas.length >= 4}
          className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50"
        >
          <Plus className="w-4 h-4" />
          Add Area ({comparisonAreas.length}/4)
        </button>
      </div>

      {/* Area Headers */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
        {comparisonAreas.map(area => (
          <div key={area.id} className="border border-gray-200 rounded-lg p-4">
            <div className="flex items-center justify-between mb-2">
              <div 
                className="w-4 h-4 rounded-full"
                style={{ backgroundColor: area.color }}
              ></div>
              <button
                onClick={() => removeArea(area.id)}
                className="text-gray-400 hover:text-red-500 transition-colors"
              >
                <X className="w-4 h-4" />
              </button>
            </div>
            <input
              type="text"
              value={area.name}
              onChange={(e) => updateAreaName(area.id, e.target.value)}
              className="w-full text-sm font-medium text-gray-900 bg-transparent border-none p-0 focus:outline-none focus:ring-0"
            />
            <div className="flex items-center gap-1 mt-1 text-xs text-gray-500">
              <MapPin className="w-3 h-3" />
              {area.properties.length} properties
            </div>
          </div>
        ))}
      </div>

      {/* Comparison Metrics */}
      <div className="space-y-6">
        {/* Average Price Comparison */}
        <div>
          <h4 className="text-sm font-medium text-gray-700 mb-3">Average Price</h4>
          <div className="space-y-2">
            {comparisonAreas.map(area => {
              const metrics = calculateMetrics(area.properties);
              const isWinner = getWinner('price') === area.id;
              const maxPrice = Math.max(...comparisonAreas.map(a => calculateMetrics(a.properties).averagePrice));
              const percentage = maxPrice > 0 ? (metrics.averagePrice / maxPrice) * 100 : 0;
              
              return (
                <div key={area.id} className="flex items-center gap-3">
                  <div 
                    className="w-3 h-3 rounded-full flex-shrink-0"
                    style={{ backgroundColor: area.color }}
                  ></div>
                  <div className="flex-1">
                    <div className="flex items-center justify-between mb-1">
                      <span className="text-sm text-gray-600">{area.name}</span>
                      <div className="flex items-center gap-1">
                        {isWinner && <TrendingUp className="w-3 h-3 text-green-600" />}
                        <span className="text-sm font-medium">{formatPrice(metrics.averagePrice)}</span>
                      </div>
                    </div>
                    <div className="w-full h-2 bg-gray-200 rounded-full overflow-hidden">
                      <div 
                        className="h-full transition-all duration-300"
                        style={{ 
                          backgroundColor: area.color, 
                          width: `${percentage}%` 
                        }}
                      ></div>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        {/* Property Density Comparison */}
        <div>
          <h4 className="text-sm font-medium text-gray-700 mb-3">Property Density</h4>
          <div className="space-y-2">
            {comparisonAreas.map(area => {
              const metrics = calculateMetrics(area.properties);
              const isWinner = getWinner('density') === area.id;
              const maxDensity = Math.max(...comparisonAreas.map(a => calculateMetrics(a.properties).density));
              const percentage = maxDensity > 0 ? (metrics.density / maxDensity) * 100 : 0;
              
              return (
                <div key={area.id} className="flex items-center gap-3">
                  <div 
                    className="w-3 h-3 rounded-full flex-shrink-0"
                    style={{ backgroundColor: area.color }}
                  ></div>
                  <div className="flex-1">
                    <div className="flex items-center justify-between mb-1">
                      <span className="text-sm text-gray-600">{area.name}</span>
                      <div className="flex items-center gap-1">
                        {isWinner && <TrendingUp className="w-3 h-3 text-green-600" />}
                        <span className="text-sm font-medium">{metrics.density} properties</span>
                      </div>
                    </div>
                    <div className="w-full h-2 bg-gray-200 rounded-full overflow-hidden">
                      <div 
                        className="h-full transition-all duration-300"
                        style={{ 
                          backgroundColor: area.color, 
                          width: `${percentage}%` 
                        }}
                      ></div>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        {/* Price per Sqft Comparison */}
        <div>
          <h4 className="text-sm font-medium text-gray-700 mb-3">Price per Sq Ft</h4>
          <div className="space-y-2">
            {comparisonAreas.map(area => {
              const metrics = calculateMetrics(area.properties);
              const isWinner = getWinner('pricePerSqft') === area.id;
              const maxPricePerSqft = Math.max(...comparisonAreas.map(a => calculateMetrics(a.properties).pricePerSqft));
              const percentage = maxPricePerSqft > 0 ? (metrics.pricePerSqft / maxPricePerSqft) * 100 : 0;
              
              return (
                <div key={area.id} className="flex items-center gap-3">
                  <div 
                    className="w-3 h-3 rounded-full flex-shrink-0"
                    style={{ backgroundColor: area.color }}
                  ></div>
                  <div className="flex-1">
                    <div className="flex items-center justify-between mb-1">
                      <span className="text-sm text-gray-600">{area.name}</span>
                      <div className="flex items-center gap-1">
                        {isWinner && <TrendingUp className="w-3 h-3 text-green-600" />}
                        <span className="text-sm font-medium">₹{Math.round(metrics.pricePerSqft).toLocaleString()}</span>
                      </div>
                    </div>
                    <div className="w-full h-2 bg-gray-200 rounded-full overflow-hidden">
                      <div 
                        className="h-full transition-all duration-300"
                        style={{ 
                          backgroundColor: area.color, 
                          width: `${percentage}%` 
                        }}
                      ></div>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        </div>
      </div>

      {isAddingArea && (
        <div className="mt-4 p-3 bg-blue-50 border border-blue-200 rounded-lg text-sm text-blue-800">
          Click and drag on the map to select an area for comparison
        </div>
      )}
    </div>
  );
}

export default AreaComparison;