import React, { useState } from 'react';
import { Filter, X, ChevronDown, ChevronUp } from 'lucide-react';

interface HeatmapFiltersProps {
  onFiltersChange: (filters: FilterState) => void;
  propertyCount: number;
}

export interface FilterState {
  propertyType: string | null;
  listingType: string | null;
  priceRange: { min: number | null; max: number | null };
  bedroomRange: { min: number | null; max: number | null };
  areaRange: { min: number | null; max: number | null };
  amenities: string[];
  ageRange: { min: number | null; max: number | null };
}

const INITIAL_FILTERS: FilterState = {
  propertyType: null,
  listingType: null,
  priceRange: { min: null, max: null },
  bedroomRange: { min: null, max: null },
  areaRange: { min: null, max: null },
  amenities: [],
  ageRange: { min: null, max: null }
};

export function HeatmapFilters({ onFiltersChange, propertyCount }: HeatmapFiltersProps) {
  const [filters, setFilters] = useState<FilterState>(INITIAL_FILTERS);
  const [isExpanded, setIsExpanded] = useState(false);
  const [activeFiltersCount, setActiveFiltersCount] = useState(0);

  const updateFilters = (newFilters: Partial<FilterState>) => {
    const updated = { ...filters, ...newFilters };
    setFilters(updated);
    
    // Count active filters
    const count = Object.entries(updated).reduce((acc, [key, value]) => {
      if (key === 'priceRange' || key === 'bedroomRange' || key === 'areaRange' || key === 'ageRange') {
        return acc + (value.min !== null || value.max !== null ? 1 : 0);
      }
      if (key === 'amenities') {
        return acc + (value.length > 0 ? 1 : 0);
      }
      return acc + (value !== null ? 1 : 0);
    }, 0);
    
    setActiveFiltersCount(count);
    onFiltersChange(updated);
  };

  const clearFilters = () => {
    setFilters(INITIAL_FILTERS);
    setActiveFiltersCount(0);
    onFiltersChange(INITIAL_FILTERS);
  };

  const propertyTypes = [
    { value: 'apartment', label: 'Apartments' },
    { value: 'villa', label: 'Villas' },
    { value: 'plot', label: 'Plots' },
    { value: 'commercial', label: 'Commercial' },
    { value: 'sea-view', label: 'Sea View' }
  ];

  const listingTypes = [
    { value: 'sell', label: 'For Sale' },
    { value: 'rent', label: 'For Rent' },
    { value: 'lease', label: 'For Lease' },
    { value: 'urgent-sale', label: 'Urgent Sale' },
    { value: 'bidding', label: 'Bidding' }
  ];

  const commonAmenities = [
    'Swimming Pool', 'Gym', 'Parking', 'Security', 'Garden',
    'Elevator', 'Power Backup', 'Water Supply', 'Club House', 'Playground'
  ];

  return (
    <div className="bg-white border border-gray-200 rounded-lg shadow-sm mb-4">
      <div 
        className="flex items-center justify-between p-4 cursor-pointer"
        onClick={() => setIsExpanded(!isExpanded)}
      >
        <div className="flex items-center gap-3">
          <Filter className="w-5 h-5 text-gray-600" />
          <span className="font-medium text-gray-900">Advanced Filters</span>
          {activeFiltersCount > 0 && (
            <span className="bg-blue-100 text-blue-800 text-xs px-2 py-1 rounded-full">
              {activeFiltersCount} active
            </span>
          )}
        </div>
        <div className="flex items-center gap-2">
          <span className="text-sm text-gray-600">{propertyCount} properties</span>
          {isExpanded ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
        </div>
      </div>

      {isExpanded && (
        <div className="border-t border-gray-200 p-4 space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {/* Property Type */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Property Type</label>
              <select
                value={filters.propertyType || ''}
                onChange={(e) => updateFilters({ propertyType: e.target.value || null })}
                className="w-full px-3 py-2 border border-gray-300 rounded-md text-sm"
              >
                <option value="">All Types</option>
                {propertyTypes.map(type => (
                  <option key={type.value} value={type.value}>{type.label}</option>
                ))}
              </select>
            </div>

            {/* Listing Type */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Listing Type</label>
              <select
                value={filters.listingType || ''}
                onChange={(e) => updateFilters({ listingType: e.target.value || null })}
                className="w-full px-3 py-2 border border-gray-300 rounded-md text-sm"
              >
                <option value="">All Listings</option>
                {listingTypes.map(type => (
                  <option key={type.value} value={type.value}>{type.label}</option>
                ))}
              </select>
            </div>

            {/* Bedrooms */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Bedrooms</label>
              <div className="flex gap-2">
                <input
                  type="number"
                  placeholder="Min"
                  value={filters.bedroomRange.min || ''}
                  onChange={(e) => updateFilters({
                    bedroomRange: { ...filters.bedroomRange, min: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-2 py-2 border border-gray-300 rounded-md text-sm"
                  min="0"
                  max="10"
                />
                <input
                  type="number"
                  placeholder="Max"
                  value={filters.bedroomRange.max || ''}
                  onChange={(e) => updateFilters({
                    bedroomRange: { ...filters.bedroomRange, max: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-2 py-2 border border-gray-300 rounded-md text-sm"
                  min="0"
                  max="10"
                />
              </div>
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {/* Price Range */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Price Range (₹)</label>
              <div className="flex gap-2">
                <input
                  type="number"
                  placeholder="Min Price"
                  value={filters.priceRange.min || ''}
                  onChange={(e) => updateFilters({
                    priceRange: { ...filters.priceRange, min: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm"
                />
                <input
                  type="number"
                  placeholder="Max Price"
                  value={filters.priceRange.max || ''}
                  onChange={(e) => updateFilters({
                    priceRange: { ...filters.priceRange, max: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm"
                />
              </div>
            </div>

            {/* Area Range */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Area (Sq Ft)</label>
              <div className="flex gap-2">
                <input
                  type="number"
                  placeholder="Min Area"
                  value={filters.areaRange.min || ''}
                  onChange={(e) => updateFilters({
                    areaRange: { ...filters.areaRange, min: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm"
                />
                <input
                  type="number"
                  placeholder="Max Area"
                  value={filters.areaRange.max || ''}
                  onChange={(e) => updateFilters({
                    areaRange: { ...filters.areaRange, max: e.target.value ? parseInt(e.target.value) : null }
                  })}
                  className="flex-1 px-3 py-2 border border-gray-300 rounded-md text-sm"
                />
              </div>
            </div>
          </div>

          {/* Amenities */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">Amenities</label>
            <div className="flex flex-wrap gap-2">
              {commonAmenities.map(amenity => (
                <button
                  key={amenity}
                  onClick={() => {
                    const newAmenities = filters.amenities.includes(amenity)
                      ? filters.amenities.filter(a => a !== amenity)
                      : [...filters.amenities, amenity];
                    updateFilters({ amenities: newAmenities });
                  }}
                  className={`px-3 py-1 text-xs rounded-full border transition-colors ${
                    filters.amenities.includes(amenity)
                      ? 'bg-blue-100 border-blue-300 text-blue-800'
                      : 'bg-gray-50 border-gray-300 text-gray-700 hover:bg-gray-100'
                  }`}
                >
                  {amenity}
                </button>
              ))}
            </div>
          </div>

          {/* Actions */}
          <div className="flex justify-between items-center pt-4 border-t border-gray-200">
            <button
              onClick={clearFilters}
              className="flex items-center gap-2 px-4 py-2 text-sm text-gray-600 hover:text-gray-800 transition-colors"
            >
              <X className="w-4 h-4" />
              Clear All Filters
            </button>
            <div className="text-sm text-gray-600">
              {propertyCount} properties match your criteria
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default HeatmapFilters;