import React, { useState, useEffect } from 'react';
import { PropertyHeatmap } from './PropertyHeatmap';
import { Property } from './types';
import { heatmapDataService } from '../../services/heatmapDataService';
import { AlertCircle, Database } from 'lucide-react';

export function HeatmapSection() {
  const [properties, setProperties] = useState<Property[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchProperties();
  }, []);

  const fetchProperties = async () => {
    try {
      setError(null);
      const result = await heatmapDataService.fetchProperties();
      
      console.log('Fetched properties result:', result);
      
      if (result.error) {
        setError(result.error);
        setProperties([]);
      } else {
        console.log('Setting properties:', result.properties.length, 'properties');
        console.log('First property:', result.properties[0]);
        setProperties(result.properties);
      }
    } catch (error) {
      console.error('Error fetching properties:', error);
      setError(error instanceof Error ? error.message : 'Failed to load property data');
      setProperties([]);
    } finally {
      setLoading(false);
    }
  };

  if (loading) {
    return (
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Market Insights</h2>
            <p className="text-gray-600">Loading real property data...</p>
          </div>
          <div className="bg-white rounded-lg p-8 shadow-sm">
            <div className="animate-pulse">
              <div className="h-96 bg-gray-200 rounded-lg"></div>
            </div>
          </div>
        </div>
      </section>
    );
  }

  if (error) {
    return (
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Market Insights</h2>
            <p className="text-gray-600">Real-time property market analysis</p>
          </div>
          
          <div className="bg-white rounded-lg p-8 shadow-sm">
            <div className="flex items-center justify-center gap-3 text-red-600 mb-4">
              <AlertCircle className="w-6 h-6" />
              <h3 className="text-lg font-semibold">Data Loading Error</h3>
            </div>
            
            <p className="text-center text-gray-600 mb-6">{error}</p>
            
            <div className="flex justify-center gap-3">
              <button
                onClick={fetchProperties}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                Retry Loading
              </button>
              
              <button
                onClick={() => window.location.reload()}
                className="px-6 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
              >
                Refresh Page
              </button>
            </div>
          </div>
        </div>
      </section>
    );
  }

  if (properties.length === 0) {
    return (
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl font-bold text-gray-900 mb-4">Market Insights</h2>
            <p className="text-gray-600">Real-time property market analysis</p>
          </div>
          
          <div className="bg-white rounded-lg p-8 shadow-sm text-center">
            <Database className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">No Property Data Available</h3>
            <p className="text-gray-600 mb-6">
              No properties with valid coordinates and pricing found in the database.
              The heatmap requires real property data to function.
            </p>
            <button
              onClick={fetchProperties}
              className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
            >
              Check Again
            </button>
          </div>
        </div>
      </section>
    );
  }

  return (
    <section className="py-16 bg-gray-50">
      <div className="container mx-auto px-4">
        <div className="text-center mb-12">
          <h2 className="text-3xl font-bold text-gray-900 mb-4">Market Insights</h2>
          <p className="text-gray-600">
            Real-time analysis of {properties.length} properties with verified coordinates and pricing
          </p>
        </div>
        
        <div className="bg-white rounded-lg shadow-sm border border-gray-200">
          <div className="p-6">
            <PropertyHeatmap properties={properties} />
          </div>
        </div>
      </div>
    </section>
  );
}