import React, { useState } from 'react';
import { Plus, Minus, Crosshair, Square, Bookmark, Download, RotateCcw } from 'lucide-react';

interface InteractiveControlsProps {
  onZoomIn: () => void;
  onZoomOut: () => void;
  onResetView: () => void;
  onAreaSelect: (mode: 'rectangle' | 'circle' | null) => void;
  onBookmarkArea: () => void;
  onExportData: () => void;
  isAreaSelectMode: 'rectangle' | 'circle' | null;
  currentZoom: number;
}

export function InteractiveControls({
  onZoomIn,
  onZoomOut,
  onResetView,
  onAreaSelect,
  onBookmarkArea,
  onExportData,
  isAreaSelectMode,
  currentZoom
}: InteractiveControlsProps) {
  const [showTooltip, setShowTooltip] = useState<string | null>(null);

  const controls = [
    {
      id: 'zoom-in',
      icon: Plus,
      label: 'Zoom In',
      onClick: onZoomIn,
      disabled: currentZoom >= 18
    },
    {
      id: 'zoom-out',
      icon: Minus,
      label: 'Zoom Out',
      onClick: onZoomOut,
      disabled: currentZoom <= 3
    },
    {
      id: 'reset-view',
      icon: RotateCcw,
      label: 'Reset View',
      onClick: onResetView,
      disabled: false
    },
    {
      id: 'select-rectangle',
      icon: Square,
      label: 'Select Rectangle Area',
      onClick: () => onAreaSelect(isAreaSelectMode === 'rectangle' ? null : 'rectangle'),
      active: isAreaSelectMode === 'rectangle'
    },
    {
      id: 'select-circle',
      icon: Crosshair,
      label: 'Select Circular Area',
      onClick: () => onAreaSelect(isAreaSelectMode === 'circle' ? null : 'circle'),
      active: isAreaSelectMode === 'circle'
    },
    {
      id: 'bookmark',
      icon: Bookmark,
      label: 'Bookmark Current View',
      onClick: onBookmarkArea,
      disabled: false
    },
    {
      id: 'export',
      icon: Download,
      label: 'Export Analysis',
      onClick: onExportData,
      disabled: false
    }
  ];

  return (
    <div className="absolute top-4 left-4 z-50 flex flex-col gap-2" style={{ position: 'absolute' }}>
      {controls.map(control => {
        const Icon = control.icon;
        return (
          <div key={control.id} className="relative">
            <button
              onClick={control.onClick}
              disabled={control.disabled}
              onMouseEnter={() => setShowTooltip(control.id)}
              onMouseLeave={() => setShowTooltip(null)}
              className={`
                w-10 h-10 rounded-lg shadow-lg border transition-all duration-200
                flex items-center justify-center
                ${control.disabled 
                  ? 'bg-gray-100 border-gray-200 text-gray-400 cursor-not-allowed' 
                  : control.active
                    ? 'bg-blue-600 border-blue-600 text-white shadow-blue-200'
                    : 'bg-white border-gray-200 text-gray-700 hover:bg-gray-50 hover:shadow-md'
                }
              `}
            >
              <Icon className="w-5 h-5" />
            </button>
            
            {showTooltip === control.id && (
              <div className="absolute left-12 top-1/2 transform -translate-y-1/2 z-60">
                <div className="bg-gray-900 text-white text-xs px-2 py-1 rounded whitespace-nowrap">
                  {control.label}
                  <div className="absolute left-0 top-1/2 transform -translate-y-1/2 -translate-x-1 w-0 h-0 border-t-2 border-b-2 border-r-2 border-transparent border-r-gray-900"></div>
                </div>
              </div>
            )}
          </div>
        );
      })}
      
      {/* Zoom Level Indicator */}
      <div className="bg-white border border-gray-200 rounded-lg shadow-lg px-3 py-2 mt-2">
        <div className="text-xs text-gray-600 text-center">
          Zoom: {currentZoom}
        </div>
      </div>
    </div>
  );
}

interface AreaSelectionOverlayProps {
  mode: 'rectangle' | 'circle' | null;
  onCancel: () => void;
}

export function AreaSelectionOverlay({ mode, onCancel }: AreaSelectionOverlayProps) {
  if (!mode) return null;

  return (
    <div className="absolute inset-0 z-40 bg-black bg-opacity-20 flex items-center justify-center">
      <div className="bg-white rounded-lg shadow-xl p-6 max-w-sm mx-4">
        <h3 className="text-lg font-semibold text-gray-900 mb-2">
          {mode === 'rectangle' ? 'Rectangle Selection' : 'Circle Selection'}
        </h3>
        <p className="text-gray-600 text-sm mb-4">
          {mode === 'rectangle' 
            ? 'Click and drag to select a rectangular area on the map'
            : 'Click on the map to select a circular area'
          }
        </p>
        <div className="flex gap-3">
          <button
            onClick={onCancel}
            className="flex-1 px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
          >
            Cancel
          </button>
        </div>
      </div>
    </div>
  );
}

export default InteractiveControls;